; Macros and Modules Made (fairly) Simple
;
; Copyright (C) 1993 Alan Bawden

(module modules
  (use scheme extensions)
  (export eval-in-module expand-module load-module module-search-rules))

; The overall strategy is to compile our extended Scheme dialect by
; generating more standard (R4RS) Scheme code.  It is not very helpful to
; think of this process as macro "expansion", because the generated code
; differs from the source code a great deal.  It is better to think of this
; process as compilation for a run-time environment that just -happens- to
; be standard R4RS Scheme.

; Each module will be compiled into a sequence of `define' statements (to
; define variables) and expressions (to be evaluated for effect).  The
; defined variables will all contain the name of the module.  Generated
; code will refer to the exported variables of other modules directly by
; name, so the normal act of evaluating (or loading) a Scheme program will
; serve to link modules together.

; Some terminology will make things easier to explain: "Forms" are the
; syntactic entities out of which programs are recursively constructed.  A
; form is any expression, any definition, or any syntactic keyword.
; Examples of forms:

;  17
;  #T
;  car
;  (+ x 4)
;  (lambda (x) x)
;  (define pi 3.14159)
;  if
;  define

; (The word "form" used to be a synonym for the word "expression" in Lisp,
; but it seems to be out of fashion nowadays; this is an attempt to revive
; an old word with a new meaning.)

; "Classification" is the process that takes a form, and decides exactly
; what kind of form it is (variable, keyword, definition, constant,
; combination, abstraction, conditional, assignment, etc.).  Note that in
; order to classify a form it is -not-, in general, necessary to recursively
; examine all of its sub-forms.  E.g., if `car' is classified as a
; variable, then `(car (f x))' can be classified as a combination without
; examining `(f x)'.

; A "syntactic environment" (usually called just an "environment" below)
; maps the symbols that appear in forms to their (syntactic) meanings.  A
; symbol will almost invariably have as its meaning either a variable or a
; keyword.

; The compiler below operates on each form in two phases.  The first phase
; is classification, it takes a form and a syntactic environment and
; returns an "item" (for want of a better name).  An item can be further
; operated upon to return the final generated code for the original form.

;
; Syntactic Environments
;

; "Environment" here means "syntactic-environment" -- that is, an
; environment that maps a symbol to its compile-time meaning.  We call the
; structure that represents this compile-time meaning an "item" (for want
; of a better word).

; Environments are built from frames in more-or-less the traditional way:

(define-structure frame
  prefix		; `prefix' and `count' are used to generate unique
  count			; names in generated code.
  defined		; A list of all symbols defined in this frame.
  entries		; A list of (<symbol> . <item>) pairs.  One entry
			; for every symbol that has ever been defined or
			; referenced in this frame.
  modules		; A list of modules used by this frame.  Ignore
			; this slot until you want to understand how
			; modules work.
  next			; The next frame, or #F.
  )

; `new-environment' builds new frames.  If this frame is the top-level
; frame in a module, `module-desc' will be its description, from which the
; `frame-prefix' is generated.  Otherwise we just assign it a number.

(define frame-prefix-counter 0)

(define (new-environment module-desc next-frame)
  (make-frame (string-append
	        "_"
		(cond (module-desc
		       (symbol->string (module-desc-name module-desc)))
		      (else
		       (set! frame-prefix-counter (+ 1 frame-prefix-counter))
		       (number->string frame-prefix-counter)))
		".")
	      0 '() '() '() next-frame))

; The basic operations on a frame's list of entries:

(define (put-frame-item env symbol item)
  ;; returns `item'
  (set-frame-entries! env
		      (cons (cons symbol item)
			    (frame-entries env)))
  item)

(define (get-frame-item env symbol)
  (cond ((assq symbol (frame-entries env)) => cdr)
	(else #F)))

; In addition to the entries in a frame, we will also need a global table
; giving some symbols an "a priori" meaning.  This is -not- a "top-level
; environment" because no symbol that actually occurs in source code will
; ever be given a meaning this way -- only generated symbols will have
; entries here.

(define a-priori-item-table '())

(define (put-a-priori-item symbol item)
  ;; returns `item'
  (set! a-priori-item-table
	(cons (cons symbol item)
	      a-priori-item-table))
  item)

(define (get-a-priori-item symbol)
  (cond ((assq symbol a-priori-item-table) => cdr)
	(else #F)))

; Environments are searched by `environment-lookup'.  Note how each frame
; remembers -all- the symbols ever looked up in that frame, not just the
; symbols defined there.  This is not done for caching, although it
; does have that effect, but so that we can insure that a symbol always
; has the same meaning in any environment.

; Also note that the modules that are used in a frame are searched -after-
; looking in the frame itself, but -before- moving on to search the next
; frame.  `search-used-modules' returns `#F' when there are no used
; modules, which is all you need to know until modules are introduced
; below.

(define (environment-lookup env symbol)
  (let recur ((env env))
    (if env
	(or (get-frame-item env symbol)
	    (put-frame-item env symbol (or (search-used-modules env symbol)
					   (recur (frame-next env)))))
	(or (get-a-priori-item symbol)
	    (put-a-priori-item symbol (make-undefined-item symbol))))))

; New environment entries are made by `environment-define'.  It first
; checks to be certain that the symbol in question hasn't already been
; defined or used in this frame.  It also maintains the `defined' slot.

(define (environment-define env symbol item)
  (when (get-frame-item env symbol)
    (if (memq symbol (frame-defined env))
	(error "Defined twice in the same scope:" symbol)
	(error "Can't handle forward reference to:" symbol)))
  (set-frame-defined! env (cons symbol (frame-defined env)))
  (put-frame-item env symbol item))

; A common operation is to make a new entry in a frame for an actual
; run-time variable.  Here we see how the `prefix' and `count' in a frame
; are used to generate a name for that variable.  (Later, it will turn out
; to be convenient for this operation to return that generated variable as
; a value.)  Note that the generated name also includes the name of the
; original symbol -- this serves no purpose other than to make it possible
; to read the generated code.

(define (environment-define-variable env symbol)
  (let ((genvar (string->symbol
		  (string-append
		    (symbol->string symbol)	; can be omitted
		    (frame-prefix env)
		    (let ((n (frame-count env)))
		      (set-frame-count! env (+ n 1))
		      (number->string n))))))
    (environment-define env symbol (make-variable-item genvar))
    genvar))

; Here is the basic mechanism that allows programmers to write macros that
; have no name-conflict problems.  `generate-meaningful-symbol' creates a
; new symbol that, a priori, has -exactly- the same meaning as a specified
; symbol in a specified environment.  We take advantage of a common Scheme
; extension, `string->uninterned-symbol', to create the new symbol with the
; same name as the old symbol, but nothing actually depends on this.  If
; all generated symbols were named `foo', nothing would break.  All that
; matters is that they all be -different- symbols (in the sense of `eq?').

(define (generate-meaningful-symbol root-symbol env)
  (let ((new-symbol (string->uninterned-symbol (symbol->string root-symbol))))
    ;; must touch it!
    (put-a-priori-item new-symbol (environment-lookup env root-symbol))
    new-symbol))

;
; Items
;

; So far, I haven't said anything much about the "meanings" stored in
; environments.  The reader will have noticed that the environment code in
; the previous section calls the functions `make-variable-item' and
; `make-undefined-item', but exactly what an "item" is has been left
; unspecified.  In addition to storing items in environments as the
; meanings of symbols, items are also returned by the fist pass
; (classification) of the compiler.  Thus, there is a different kind of
; item for each kind of basic form in the language.

(define-structure item
  kind			; One of: `variable', `keyword', `sequence',
			; `definition', `constant', `combination',
			; `abstraction', `conditional', `assignment',
			; `delay', or `undefined'.  Only the first three
			; are actually tested for.
  data			; If the kind is `keyword', the data is a procedure
			; to be used to process a form that uses the keyword.
			; If the kind is `variable', the data is the
			; generated variable to be used in generated code.
			; If the kind is `definition', the data is the
			; generated variable that is to be assigned in the
			; generated code.
			; If the kind is `sequence', the data is a list of
			; items that make up the sequence (none of which
			; can be a sequence itself).
			; In all other cases the data is `#F'.
  codegen		; A procedure to be called to generate the code for
			; this item.
  )

; The second phase of compilation simply asks the item to finish the job of
; compiling itself.

(define (codegen item) ((item-codegen item)))

; The reason for the delay in finishing the job is to avoid examining any
; sub-forms of the original until as many lexically relevant definitions as
; possible have been discovered.  This technique permits forward references
; to macro definitions to work properly in most common situations, thus
; minimizing dependence on the order in which forms are written.

; `classify' is pretty simple.  Symbols are simply looked up in the syntactic
; environment.  For a pair, the `car' is recursively classified, and if it is
; a keyword, then the `item-data' is a procedure that can finish the job of
; classification, otherwise the form is a combination.  All other forms are
; constants.

; The boolean flag `def?' is `#T' if it is permissible for the form to
; -add- new definitions to the syntactic environment.  Most calls to
; `classify' supply `#F'.

(define (classify form env def?)
  (cond ((symbol? form) (environment-lookup env form))
	((pair? form)
	 (let ((op-item (classify (car form) env #F)))
	   (if (eq? (item-kind op-item) 'keyword)
	       ((item-data op-item) form env def?)
	       (make-combination-item op-item form env))))
	(else (make-constant-item form))))

; Two common combinations of `classify' and `codegen':

(define (compile form env)
  (codegen (classify form env #F)))

(define (compile-list forms env)
  (map (lambda (form) (compile form env)) forms))

; Another common case is to classify a sequence of forms in the body of
; some other form, such as `lambda' or `begin'.  Because of the case where
; definitions are allowed (i.e., `def?' is `#T'), we always perform this
; operation in left-to-right order.  This helps us find "semantic fixed
; points" in more intuitive situations.  Also, code generation depends on
; the variables defined in a module being defined in a predictable,
; repeatable order.  `classify-body' also flattens out any nested sequences
; it finds, so that we can more easily find all the definitions that were
; made.

(define (classify-body forms env def?)
  (let loop ((forms forms))
    (if (null? forms)
	'()
	(let ((item (classify (car forms) env def?)))
	  (if (eq? (item-kind item) 'sequence)
	      (append (item-data item) (loop (cdr forms)))
	      (cons item (loop (cdr forms))))))))

; Now come constructors for the items themselves.  The first few are pretty
; straightforward given the explanations above, and given that code
; generation only involves generating Scheme.

(define (make-undefined-item symbol)
  (make-item 'undefined #F
    (lambda ()
      (error "Undefined:" symbol))))

(define (make-variable-item genvar)
  (make-item 'variable genvar
    (lambda () genvar)))

(define (make-combination-item op-item form env)
  (make-item 'combination #F
    (lambda ()
      `(,(codegen op-item)
	,@(compile-list (cdr form) env)))))

(define (make-constant-item datum)
  (make-item 'constant #F
    (lambda ()
      `(quote ,datum))))

(define (make-sequence-item items)
  (make-item 'sequence items
    (lambda ()
      (cond ((null? items) `#F)
	    ((null? (cdr items)) (codegen (car items)))
	    (else `(begin ,@(map codegen items)))))))

;
; Primitive Keywords
;

; Now we must construct the items that will be the meanings of the standard
; primitive keywords such as `quote', `lambda', `if', `set!', etc.  If such
; an item is every called upon to generate code for itself, then the user
; has made an error.  The only interesting thing about a keyword is how you
; classify a form that uses that keyword:

(define (make-keyword-item classify)
  (make-item 'keyword classify
    (lambda ()
      (error "Keyword used as an expression."))))

; Variables whose names begin with "`$'" will be used to contain the items
; for the primitive keywords:

(define $quote
  (make-keyword-item
    (lambda (form env def?)
      (make-constant-item (cadr form)))))

(define $set!
  (make-keyword-item
    (lambda (form env def?)
      (make-item 'assignment #F
	(lambda ()
	  (let ((item (classify (cadr form) env #F)))
	    (unless (eq? (item-kind item) 'variable)
	      (error "Illegal assignment:" form))
	    `(set! ,(item-data item) ,(compile (caddr form) env))))))))

(define $if
  (make-keyword-item
    (lambda (form env def?)
      (make-item 'conditional #F
	(lambda ()
	  `(if ,@(compile-list (cdr form) env)))))))

; Note how `begin' passes the `def?' flag down to the call to
; `classify-body'.  Combined with the fact that `classify-body' flattens
; out nested sequences, this makes `begin' transparent to definitions, as
; required by R4RS.

(define $begin
  (make-keyword-item
    (lambda (form env def?)
      (make-sequence-item (classify-body (cdr form) env def?)))))

; Ignore this if you don't care (-!-)

(define $delay
  (make-keyword-item
    (lambda (form env def?)
      (make-item 'delay #F
	(lambda ()
	  `(delay ,@(compile-list (cdr form) env)))))))

; Code generation for an abstraction is the first time we have encountered
; anything that actually constructs a new environment frame.  The auxiliary
; procedure `compile-bvl' simply walks over the formal parameters
; (left-to-right) and makes an entry in the new environment for each of
; them.  It returns a congruent formal parameter list for the generated
; code.  Another auxiliary procedure, `compile-body' handles the allocation
; of a second environment for any internal `define's.  `compile-body' also
; contains the first call to classify we have seen that explicily passes
; `#T' in order to permit definitions.

(define $lambda
  (make-keyword-item
    (lambda (form env def?)
      (make-item 'abstraction #F
	(lambda ()
	  (let* ((env (new-environment #F env))
		 (genvars (compile-bvl (cadr form) env)))
	    `(lambda ,genvars
	       ,@(compile-body (cddr form) env))))))))

(define (compile-bvl bvl env)
  (let recur ((bvl bvl))
    (cond ((symbol? bvl)
	   (environment-define-variable env bvl))
	  ((pair? bvl)
	   (let ((x (recur (car bvl))))		; left-to-right!
	     (cons x (recur (cdr bvl)))))
	  (else bvl))))

(define (compile-body forms env)
  (map codegen (classify-body forms (new-environment #F env) #T)))

; `simple-define' is a version of `define' that doesn't support the
; inessential features of the R4RS `define' syntax.  It does, however have
; a new behavior that permits it to be used to define symbols as keywords:
; If the sub-form classifies as a keyword, then we enter the symbol to
; be defined into the environment as an alias for the same keyword item,
; and we vanish into an empty sequence (because no code will need to be
; generated).  Otherwise, we make an entry in the environment for a new
; variable, and return a definition item that will generate the appropriate
; code to initialize the variable at run-time.

; Later we will define a full-featured `define' as a macro that expands
; into a use of `simple-define'.  It will inherit this alias-making
; behavior so that it too can be used to define keywords.  Also we will be
; able to use the module system to hide the definition of `simple-define'
; from ordinary users.

(define $simple-define
  (make-keyword-item
    (lambda (form env def?)
      (unless def?
	(error "Misplaced definition:" form))
      (let ((symbol (cadr form))
	    (item (classify (caddr form) env #F)))
	(cond ((eq? (item-kind item) 'keyword)
	       (environment-define env symbol item)
	       $empty-sequence)
	      (else
	       (let ((genvar (environment-define-variable env symbol)))
		 (make-item 'definition genvar
		   (lambda ()
		     `(define ,genvar ,(codegen item)))))))))))

(define $empty-sequence (make-sequence-item '()))

; Another non-standard primitive keyword that we will want to hide from
; ordinary users is `define-alias', which can be used to name essentially
; arbitrary items in syntactic environments.  It will prove useful in
; building the initial `scheme' module.

(define $define-alias
  (make-keyword-item
    (lambda (form env def?)
      (unless def?
	(error "Misplaced definition:" form))
      (environment-define env
			  (cadr form)
			  (classify (caddr form) env #F))
      $empty-sequence)))

; Finally, here's how we let the user create new keywords.  The details of
; `make-transformer' will be given later, and really needn't concern us
; just yet.  Somehow the body of a `macro' form will be used to generate a
; "transformer", a procedure that accepts a form, and a "gensym" procedure,
; and returns a new form to be compiled in the old form's place.  The
; "gensym" procedure can be used by the transformer to generate symbols
; that have the same meaning as specified symbols have in the syntactic
; environment where the `macro' form was written:

(define $macro
  (make-keyword-item
    (lambda (form env def?)
      (let ((transformer (make-transformer (cadr form)	; -!- early???
					   (caddr form)
					   (cdddr form)))
	    (gensym (lambda (symbol)
		      (generate-meaningful-symbol symbol env))))
	(make-keyword-item
	  (lambda (form env def?)
	    (classify (transformer form gensym) env def?)))))))

;
; Modules
;

; Now we proceed to explore the simple module system.  We only support a
; flat namespace of modules, named by symbols.  Each module known to the
; system is described by a `module-desc' structure

(define-structure module-desc
  name			; Every module has a unique name.  A symbol.
  exports		; A list of all the symbols that are exported from
			; this module.
  initial-use		; A list of names of other modules whose exports
			; are to be imported into this module for use in
			; the body.  If this list is empty, the body will
			; be useless, since all names will be undefined.
  body			; A list of all the forms that comprise the body of
			; the module.
  env			; The top-level frame for this module, or #F if it
			; hasn't been created yet.
  items			; The results of classifying the body of the
			; module, if that has been done.
  )

; All the module descriptions are kept in a table keyed by their names:

(define module-desc-table '())

(define (put-module name exports initial-use body)
  ;; returns the new module-desc
  (let ((mdesc (make-module-desc name exports initial-use body #F #F)))
    (set! module-desc-table
	  (cons (cons name mdesc)
		module-desc-table))
    mdesc))

(define (get-module-desc name)
  (cond ((assq name module-desc-table) => cdr)
	(else #F)))

; `create-module-desc' is the usual interface to `put-module'.  It parses
; the standard way modules are written in files as a header of the form
; `(module name option ...)' followed by a sequence of forms comprising the
; body.  Two options are understood, `(export symbol ...)', which simply
; adds the symbols to this modules export list, and `(use name ...)',
; which add the module names to this modules list of modules to use.

(define (create-module-desc header&body)
  (let* ((header (car header&body))
	 (body (cdr header&body))
	 (name (cadr header)))
    (let loop ((specs (cddr header))
	       (exports '())
	       (initial-use '()))
      (cond ((null? specs)
	     (put-module name exports initial-use body))
	    ((eq? (caar specs) 'export)
	     (loop (cdr specs) (append (cdar specs) exports) initial-use))
	    ((eq? (caar specs) 'use)
	     (loop (cdr specs) exports (append (cdar specs) initial-use)))
	    (else (loop (cdr specs) exports initial-use))))))

; `find-module' checks the table of already seen modules, if it doesn't
; find it, it give a sequence of "search rules" a chance to create the
; module before `find-module' signals an error.

(define (find-module-desc name)
  (or (get-module-desc name)
      (let loop ((rules module-search-rules))
	(cond ((null? rules) (error "Can't find module:" name))
	      (((car rules) name) => create-module-desc)
	      (else (loop (cdr rules)))))))

; Currently, the only rule we have looks for the module in the current
; working directory.

(define module-search-rules
  (list (lambda (name)
	  (let ((pathname (string-append (list->string
					   (map char-downcase
						(string->list
						  (symbol->string name))))
					 ".scm")))
	    (and (file-exists? pathname)
		 (call-with-input-file pathname
		   (lambda (port)
		     (let ((header (read port)))
		       (if (or (eof-object? header)
			       (not (module-header? name header)))
			   #F
			   (cons header
				 (let loop ()
				   (let ((x (read port)))
				     (if (eof-object? x)
					 '()
					 (cons x (loop)))))))))))))))

(define (module-header? name x)
  (and (pair? x)
       (pair? (cdr x))
       (eq? (car x) 'module)
       (eq? (cadr x) name)))

; So far this has just been a lot of mechanism to locate modules and
; associate a description with them.  nothing has actually looked at the
; body of the module yet.  `find-module-env' is what actually gets the ball
; rolling.  If the module already as an environment, this simply returns
; it.  If not, it creates an environment, sets up the initial list of other
; modules to use, and then classifies the body of the module.  This
; extracts all of the top-level definitions.  The resulting list of items
; is saved for later (in case we actually want to generate code for this
; module).

(define (find-module-env mdesc)
  (or (module-desc-env mdesc)
      (let ((env (new-environment mdesc #F)))
	(set-module-desc-env! mdesc env)	; -before- we start!
	(environment-use-module-list env (module-desc-initial-use mdesc))
	(set-module-desc-items! mdesc
				(classify-body (module-desc-body mdesc)
					       env
					       #T))
	env)))

; A common case is to want to find the meaning of a particular symbol at
; top-level in a particular module:

(define (module-desc-lookup mdesc symbol)
  (environment-lookup (find-module-env mdesc) symbol))

; Most of the hair in the following two procedures is really just error
; checking.  The goal is simply to add a new module to the list of module
; descriptions used by a particular environment frame.  The error checking
; is to guard against the case where using a new module introduces a name
; that has already been assumed to have an incompatible meaning.

(define (environment-use-module-list env names)
  (for-each (lambda (name)
	      (environment-use-module env name))
	    names))

(define (environment-use-module env name)
  (let ((defined (frame-defined env))
	(mdescs (frame-modules env))
	(new-mdesc (find-module-desc name)))
    (unless (memq new-mdesc mdescs)
      (for-each (lambda (symbol)
		  (let ((item (get-frame-item env symbol)))
		    (when (and item
			       (not (memq symbol defined))
			       (not (eq? item
					 (module-desc-lookup new-mdesc
							     symbol))))
		      (error "Can't handle forward reference to exported:"
			     symbol))))
		(module-desc-exports new-mdesc))
      (set-frame-modules! env (cons new-mdesc mdescs)))))

; This is the procedure that I told you to ignore way back in the
; definition of `environment-lookup'.  It simply searches all the modules
; used by the given environment frame looking for a possible export.  The
; only complexity is again an error check.

(define (search-used-modules env symbol)
  (define (search mdescs)
    (cond ((null? mdescs) #F)
  	  ((memq symbol (module-desc-exports (car mdescs)))
	   (let ((item1 (module-desc-lookup (car mdescs) symbol))
		 (item2 (search (cdr mdescs))))
	     (when (and item2 (not (eq? item2 item1)))
	       (error "Incompatibly exported from multiple modules:" symbol))
	     item1))
	  (else
	   (search (cdr mdescs)))))
  (search (frame-modules env)))

; Two new keywords are introduced for the benefit of modules.  `use' has
; the same effect on an arbitrary environment that the `use' option in a
; module header has.

(define $use
  (make-keyword-item
    (lambda (form env def?)
      (unless def?
	(error "Misplaced:" form))
      (environment-use-module-list env (cdr form))
      $empty-sequence)))

; `(access module-name symbol)' can be used to access a module's exported
; symbols without having to `use' the entire module.

(define $access
  (make-keyword-item
    (lambda (form env def?)
      (let ((mdesc (find-module-desc (cadr form)))
	    (symbol (caddr form)))
	(unless (memq symbol (module-desc-exports mdesc))
	  (error "Not exported:" form))
	(environment-lookup (find-module-env mdesc) (caddr form))))))

;
; The Primitive-Keywords Module
;

; There are two built-in modules.  The `primitive-keywords' module contains
; just the definitions of the primitive keywords defined herein.  The
; `user' module exists for the benefit of a simple read-eval-print loop.

; In addition to the primitive keywords, we need a way to get at the
; primitive procedure names in the underlying Scheme implementation.  
; `(primitive-access symbol)' creates an item that directly compiles into
; the given variable name.

(define $primitive-access
  (make-keyword-item
    (lambda (form env def?)
      (make-variable-item (cadr form)))))

; Now we have defined the last of the keywords.  This procedure initializes
; a few things and then builds the built-in modules.

(define (initialize-modules)
  (set! a-priori-item-table '())
  (set! module-desc-table '())
  ;; set up `user':
  (put-module 'user '() '(scheme) '())
  ;; set up `primitive-keywords':
  (let* ((mdesc (put-module 'primitive-keywords
			    '(access begin define-alias delay if lambda
			      macro primitive-access quote set!
			      simple-define use)
			    '() '()))
	 (env (new-environment mdesc #F))
	 (def (lambda (symbol item)
		(environment-define env symbol item))))
    (set-module-desc-env! mdesc env)
    (def 'access $access)
    (def 'begin $begin)
    (def 'define-alias $define-alias)
    (def 'delay $delay)
    (def 'if $if)
    (def 'lambda $lambda)
    (def 'macro $macro)
    (def 'primitive-access $primitive-access)
    (def 'quote $quote)
    (def 'set! $set!)
    (def 'simple-define $simple-define)
    (def 'use $use)
    (set-module-desc-items! mdesc '())))

;
; Evaluation
;

; Finally, here is the glue that let's get at the underlying Scheme
; implementation.  `system-eval' should be defined so as to evaluate
; expressions in the same environment that `load' uses.

(define (system-eval form)
  (eval form user-initial-environment))

; `make-transformer' is one of the first in line waiting for `system-eval'.
; Notice that the call to `new-environment' specifies neither an associated
; module, or a containing environment frame.  All we have is a list of
; module name for it to use.  In effect, each macro body is a little
; anonymous module.

; -!- Break glass in case of emergency:
; (define (make-transformer bvl module-names body)
;   (system-eval `(lambda ,bvl ,@body)))

(define (make-transformer bvl module-names body)
  (let* ((env (new-environment #F #F))
	 (genvars (compile-bvl bvl env)))
    (environment-use-module-list env module-names)
    (system-eval `(lambda ,genvars
		    ,@(compile-body body env)))))

; The rest of the uses for `system-eval' require is to finally finish the
; job of generating code for the body of a module.  `codegen-module' does
; this by first calling `find-module-env' to insure that the body has been
; classified into a list of items, and then generating code for each item.

(define (codegen-module module-name)
  (let ((mdesc (find-module-desc module-name)))
    (find-module-env mdesc)
    (map codegen (module-desc-items mdesc))))

; One thing we can do now is load the module up and use it:

(define (load-module module-name)
  (for-each system-eval (codegen-module module-name)))

; Another thing we can do is save the generated code for later in a file,
; and load it using `load':

(define (expand-module module-name pathname)
  (let ((code (codegen-module module-name)))
    (call-with-output-file pathname
      (lambda (port)
	(define (out x) (write x port) (newline port))
	(out `(quote (expanded module ,module-name)))
	(for-each out code)))))

; Finally, it is convenient to have our own version of `eval' that lets us
; evaluate arbitrary forms in any module.  

(define (eval-in-module form module-name)
  (system-eval
    (codegen
      (classify form
		(find-module-env
		  (find-module-desc module-name))
		#T))))

(define (rep module-name)
  (let loop ()
    (newline)
    (let ((form (read)))
      (unless (eof-object? form)
	(let ((value (eval-in-module form module-name)))
	  (newline)
	  (write value)
	  (loop))))))

(initialize-modules)
(rep 'user)

; Local Variables:
; mode: Scheme
; eval: (put 'make-item 'scheme-indent-hook 2)
; eval: (put 'make-keyword-item 'scheme-indent-hook 0)
; End:
