; Macros and Modules Made Simple
;
; Copyright (C) 1993 Alan Bawden

(module scheme
  ;; This module defines the standard R4RS names, plus five new keywords
  ;; that are generally useful when working with macros and modules.

  ;; We import only the primitive keywords defined in `modules.scm'.  Some
  ;; of these will be directly exported to our clients (such as `quote' and
  ;; `set!').  Others will be used just to help define new keywords that
  ;; will be exported (e.g., `define' will be defined in terms of
  ;; `simple-define'):

  (use primitive-keywords)

  ;; The R4RS procedure names:

  (export * + - / < <= = > >= abs acos angle append apply asin assoc assq
	  assv atan boolean? caaaar caaadr caaar caadar caaddr caadr caar
	  cadaar cadadr cadar caddar cadddr caddr cadr
	  call-with-current-continuation call-with-input-file
	  call-with-output-file car cdaaar cdaadr cdaar cdadar cdaddr cdadr
	  cdar cddaar cddadr cddar cdddar cddddr cdddr cddr cdr ceiling
	  char->integer char-alphabetic? char-ci<=? char-ci<? char-ci=?
	  char-ci>=? char-ci>? char-downcase char-lower-case? char-numeric?
	  char-ready? char-upcase char-upper-case? char-whitespace? char<=?
	  char<? char=? char>=? char>? char? close-input-port
	  close-output-port complex? cons cos current-input-port
	  current-output-port denominator display eof-object? eq? equal?
	  eqv? even? exact->inexact exact? exp expt floor for-each force
	  gcd imag-part inexact->exact inexact? input-port? integer->char
	  integer? lcm length list list->string list->vector list-ref
	  list-tail list? load log magnitude make-polar make-rectangular
	  make-string make-vector map max member memq memv min modulo
	  negative? newline not null? number->string number? numerator odd?
	  open-input-file open-output-file output-port? pair? peek-char
	  positive? procedure? quotient rational? rationalize read
	  read-char real-part real? remainder reverse round set-car!
	  set-cdr! sin sqrt string->list string->number string->symbol
	  string-append string-ci<=? string-ci<? string-ci=? string-ci>=?
	  string-ci>? string-copy string-fill! string-length string-ref
	  string-set! string<=? string<? string=? string>=? string>?
	  string? substring symbol->string symbol? tan transcript-off
	  transcript-on truncate vector vector->list vector-fill!
	  vector-length vector-ref vector-set! vector? with-input-from-file
	  with-output-to-file write write-char zero?)

  ;; The R4RS keywords:

  (export and begin case cond define delay do if lambda let let* letrec or
	  quasiquote quote set!)

  ;; Five new keywords for macros and modules:

  (export access define-alias defmacro macro use)

  ) ; end of (module scheme ...)

; Initially, the only usable names exported from the `scheme' module are
; those keywords that happen to be imported from `primitive-keywords' and
; directly exported.  Our first step is to get all the procedure names
; working as well.  We could do this by writing a large number of
; expressions such as:
;
;   (define-alias * (primitive-access *))
;   (define-alias + (primitive-access +))
;   ...
;   (define-alias zero? (primitive-access zero?)))
;
; But producing such repetitive code looks like an ideal job for a macro!
; (Which thus has to be written using -only- the `primitive-keywords'
; module.)

((macro (form gensym) (primitive-keywords)
   ((primitive-access cons)
    'begin
    ((primitive-access map)
     (lambda (symbol)
       ((primitive-access list)
	'define-alias
	symbol
	((primitive-access list)
	 'primitive-access
	 symbol)))
     ((primitive-access cdr) form))))
 * + - / < <= = > >= abs acos angle append apply asin assoc assq
 assv atan boolean? caaaar caaadr caaar caadar caaddr caadr caar
 cadaar cadadr cadar caddar cadddr caddr cadr
 call-with-current-continuation call-with-input-file
 call-with-output-file car cdaaar cdaadr cdaar cdadar cdaddr cdadr
 cdar cddaar cddadr cddar cdddar cddddr cdddr cddr cdr ceiling
 char->integer char-alphabetic? char-ci<=? char-ci<? char-ci=?
 char-ci>=? char-ci>? char-downcase char-lower-case? char-numeric?
 char-ready? char-upcase char-upper-case? char-whitespace? char<=?
 char<? char=? char>=? char>? char? close-input-port
 close-output-port complex? cons cos current-input-port
 current-output-port denominator display eof-object? eq? equal?
 eqv? even? exact->inexact exact? exp expt floor for-each force
 gcd imag-part inexact->exact inexact? input-port? integer->char
 integer? lcm length list list->string list->vector list-ref
 list-tail list? load log magnitude make-polar make-rectangular
 make-string make-vector map max member memq memv min modulo
 negative? newline not null? number->string number? numerator odd?
 open-input-file open-output-file output-port? pair? peek-char
 positive? procedure? quotient rational? rationalize read
 read-char real-part real? remainder reverse round set-car!
 set-cdr! sin sqrt string->list string->number string->symbol
 string-append string-ci<=? string-ci<? string-ci=? string-ci>=?
 string-ci>? string-copy string-fill! string-length string-ref
 string-set! string<=? string<? string=? string>=? string>?
 string? substring symbol->string symbol? tan transcript-off
 transcript-on truncate vector vector->list vector-fill!
 vector-length vector-ref vector-set! vector? with-input-from-file
 with-output-to-file write write-char zero?
 )

; We define `define' right away.  Note that for the first time we are using
; the `scheme' module itself in the body of this macro.  Thus we must be
; careful to only use names that are already working.  (But if we screw up,
; the system will tell us we have a problem -- it won't just silently do
; something wrong.)

(simple-define define
  (macro (form gensym)
      (scheme)		; see comment above
    (if (pair? (cadr form))
	(list (gensym 'define)
	      (caadr form)
	      (cons (gensym 'lambda) (cons (cdadr form) (cddr form))))
	(cons (gensym 'simple-define) (cdr form)))))

; Now define the `let' macro.  Since we haven't defined `quasiquote' yet,
; the definition of `let' is a bit verbose:

(define let
  (macro (form gensym) (scheme)
    (if (symbol? (cadr form))
	(list (gensym 'letrec)
	      (list (list (cadr form)
			  (cons (gensym 'lambda)
				(cons (map car (caddr form))
				      (cdddr form)))))
	      (cons (cadr form)
		    (map cadr (caddr form))))
	(cons (cons (gensym 'lambda)
		    (cons (map car (cadr form))
			  (cddr form)))
	      (map cadr (cadr form))))))

; With the addition of `let' and `define' we can now define `quasiquote':
; (It would be nice to be able to use `cond' here, but writing `cond'
; without using `quasiquote' is much harder than writing `quasiquote'
; without using `cond'!)

(define quasiquote
  (macro (form gensym) (scheme)
    (define (tag? tag x)
      (if (pair? x)
	  (if (eq? tag (car x))
	      (if (pair? (cdr x))
		  (null? (cddr x))
		  #F)
	      #F)
	  #F))
    (let ((=quote (gensym 'quote))
	  (=cons (gensym 'cons))
	  (=append (gensym 'append))
	  (=list->vector (gensym 'list->vector)))
      (define (walk template)
	(if (vector? template)
	    (list =list->vector (walk (vector->list template)))
	    (if (not (pair? template))
		(list =quote template)
		(if (tag? 'unquote template)
		    (cadr template)
		    (if (tag? 'quasiquote template)
			(walk (walk (cadr template)))
			(if (tag? 'unquote-splicing (car template))
			    (list =append
				  (cadar template)
				  (walk (cdr template)))
			    (list =cons
				  (walk (car template))
				  (walk (cdr template)))))))))
      (walk (cadr form)))))

; Now we're ready to define a simple `defmacro'.
; Syntax: (defmacro (<name> . <pattern>) (<genvar> ...) . <body>)
; Features:
;  - The input form is "destructured" according to <pattern>.
;  - The body is an ordinary Scheme program.
;  - The variables `form' and `gensym' are available for use in the body.
;  - The body always uses the `scheme' module.
;  - For each <genvar>, a new symbol is generated using `gensym', and
;  - that symbol is made the value of a variable whose name starts with "=".
; The results of using this `defmacro' are highly suggestive of better
; macro defining technology -- particularly the frequent occurance of the
; sequence ",=" in macro bodies.

(define defmacro
  (macro (form gensym) (scheme)
    (define (make-bindings pattern expr)
      (if (symbol? pattern)
	  `((,pattern ,expr))
	  (if (pair? pattern)
	      `(,@(make-bindings (car pattern) `(car ,expr))
		,@(make-bindings (cdr pattern) `(cdr ,expr)))
	      `())))
    (let ((=define (gensym 'define))
          (=macro (gensym 'macro))
          (name (caadr form))
          (pattern (cdadr form))
          (genvars (caddr form))
          (body (cdddr form)))
      `(,=define ,name
         (,=macro (form gensym) (scheme)
           (let (,@(map (lambda (name)
                          `(,(string->symbol
			       (string-append
				 "="
				 (symbol->string name)))
			    (gensym ',name)))
                        genvars)
                 ,@(make-bindings pattern `(cdr form)))
             ,@body))))))

; From here on in it's smooth sailing.  For clarity, I have ordered these
; definitions so that macros continue to be defined before they are used in
; subsequent macro bodies, but that isn't necessary -- the definitions
; below can actually be written in any order.

(defmacro (letrec specs . body) (lambda define)
  `((,=lambda ()
      ,@(map (lambda (spec)
	       `(,=define ,(car spec) ,(cadr spec)))
	     specs)
      ,@body)))

(defmacro (let* specs . body) (lambda let*)
  (if (null? specs)
      `((,=lambda () ,@body))
      (if (null? (cdr specs))
	  `((,=lambda (,(caar specs))
	       ,@body)
	    ,(cadar specs))
	  `((,=lambda (,(caar specs))
	      (,=let* ,(cdr specs)
		,@body))
	    ,(cadar specs)))))

(defmacro (and . rest) (if and)
  (if (null? rest)
      `#T
      (if (null? (cdr rest))
	  (car rest)
	  `(,=if ,(car rest)
		 (,=and ,@(cdr rest))
		 #F))))

(defmacro (or . rest) (if or let temp)
  (if (null? rest)
      `#F
      (if (null? (cdr rest))
	  (car rest)
	  `(,=let ((,=temp ,(car rest)))
	     (,=if ,=temp
		   ,=temp
		   (,=or ,@(cdr rest)))))))

(defmacro (cond clause . more) (begin if or let temp)
  ;; Note that it -is- safe to use the same TEMP everywhere:
  (let loop ((clause clause)
	     (more more))
    (if (null? more)
	(if (eq? (car clause) 'else)
	    `(,=begin ,@(cdr clause))
	    (if (null? (cdr clause))
		(car clause)
		(if (and (eq? (cadr clause) '=>) (= 3 (length clause)))
		    `(,=let ((,=temp ,(car clause)))
		       (,=if ,=temp
			     (,(caddr clause) ,=temp)))
		    `(,=if ,(car clause)
			   (,=begin ,@(cdr clause))))))
	(if (null? (cdr clause))
	    `(,=or ,(car clause) ,(loop (car more) (cdr more)))
	    (if (and (eq? (cadr clause) '=>) (= 3 (length clause)))
		`(,=let ((,=temp ,(car clause)))
		   (,=if ,=temp
			 (,(caddr clause) ,=temp)
			 ,(loop (car more) (cdr more))))
		`(,=if ,(car clause)
		       (,=begin ,@(cdr clause))
		       ,(loop (car more) (cdr more))))))))

(defmacro (case thing clause . more) (let temp begin if memq quote)
  `(,=let ((,=temp ,thing))
     ,(let loop ((clause clause)
		 (more more))
	(if (null? more)
	    (cond ((eq? (car clause) 'else)
		   `(,=begin ,@(cdr clause)))
		  (else
		   `(,=if (,=memq ,=temp (,=quote ,(car clause)))
			  (,=begin ,@(cdr clause)))))
	    `(,=if (,=memq ,=temp (,=quote ,(car clause)))
		   (,=begin ,@(cdr clause))
		   ,(loop (car more) (cdr more)))))))

(defmacro (do specs (test . finish) . body) (let loop if begin)
  `(,=let ,=loop ,(map (lambda (spec)
			 `(,(car spec) ,(cadr spec)))
		       specs)
     (,=if ,test
	   ,(if (null? finish)
		`#F
		`(,=begin ,@finish))
	   (,=begin ,@body
		    (,=loop ,@(map caddr specs))))))

; Local Variables:
; mode: Scheme
; eval: (put 'simple-define 'scheme-indent-hook 1)
; eval: (put 'macro 'scheme-indent-hook 2)
; End:
