import re
from env.state import ProofState
from env.language import Var, And, Implies, Or, PropFalse


# A simple recursive descent parser for the proposition language.
def parse_prop(s: str):
    s = s.strip()

    # Handle parentheses
    if s.startswith("(") and s.endswith(")"):
        # Check if the parentheses are matching
        balance = 0
        for i, char in enumerate(s):
            if char == "(":
                balance += 1
            elif char == ")":
                balance -= 1
            if balance == 0 and i < len(s) - 1:
                break
        else:
            return parse_prop(s[1:-1])

    # Handle operators by precedence (lowest to highest)
    balance = 0
    for i in range(len(s) - 1, -1, -1):
        if s[i] == ")":
            balance += 1
        elif s[i] == "(":
            balance -= 1

        if balance == 0:
            if s[i : i + 2] == "->":
                return Implies(parse_prop(s[:i]), parse_prop(s[i + 2 :]))

    balance = 0
    for i in range(len(s) - 1, -1, -1):
        if s[i] == ")":
            balance += 1
        elif s[i] == "(":
            balance -= 1

        if balance == 0:
            if s[i : i + 2] == "\\/":
                return Or(parse_prop(s[:i]), parse_prop(s[i + 2 :]))
            if s[i : i + 2] == "/\\":
                return And(parse_prop(s[:i]), parse_prop(s[i + 2 :]))

    if s == "False":
        return PropFalse()

    return Var(s)


def parse_theorem_signature(theorem_string: str) -> ProofState:
    m = re.match(
        r"theorem\s+(?P<name>\w+)\s*\((?P<hyps>.*)\)\s*:\s*(?P<goal>.*)", theorem_string
    )
    if not m:
        # No hypotheses
        m = re.match(r"theorem\s+(?P<name>\w+)\s*:\s*(?P<goal>.*)", theorem_string)
        if not m:
            raise ValueError(f"Could not parse theorem signature: {theorem_string}")
        hyps_str = ""
    else:
        hyps_str = m.group("hyps")

    goal_str = m.group("goal")
    goal = parse_prop(goal_str)

    hyps = {}
    if hyps_str:
        # This regex handles splitting hypotheses correctly
        hyp_strs = re.findall(r"[^,]+:[^,]+(?:\(.*\))?", hyps_str)
        for hyp_str in hyp_strs:
            name, prop_str = hyp_str.split(":", 1)
            hyps[name.strip()] = parse_prop(prop_str)

    return ProofState(goals=[goal], context=hyps)


def parse_theorem_with_proof(theorem_string: str) -> tuple[ProofState, list[str]]:
    """
    Parses a full theorem string (signature and proof) into an initial
    ProofState and a list of tactics.
    """
    if ":=" not in theorem_string:
        raise ValueError("Theorem string does not contain a proof (':=').")

    sig_part, proof_part = theorem_string.split(":=", 1)
    sig_part = sig_part.strip()
    proof_part = proof_part.strip()

    initial_state = parse_theorem_signature(sig_part)

    tactics = [line.strip() for line in proof_part.split("\n") if line.strip()]

    return initial_state, tactics

    tactics = [line.strip() for line in proof_part.split("\n") if line.strip()]

    return initial_state, tactics

