class Prop:
    """Base class for all propositions."""

    args: tuple

    def __eq__(self, other):
        return isinstance(other, self.__class__) and self.args == other.args

    def __hash__(self):
        return hash((self.__class__.__name__, self.args))


class Var(Prop):
    """A propositional variable, e.g., 'p'."""

    def __init__(self, name: str):
        self.name = name
        self.args = (name,)

    def __repr__(self) -> str:
        return self.name


class And(Prop):
    """A logical conjunction (P /\\ Q)."""

    def __init__(self, left: Prop, right: Prop):
        self.left = left
        self.right = right
        self.args = (left, right)

    def __repr__(self) -> str:
        return f"({self.left} /\\ {self.right})"


class Implies(Prop):
    """A logical implication (P -> Q)."""

    def __init__(self, antecedent: Prop, consequent: Prop):
        self.antecedent = antecedent
        self.consequent = consequent
        self.args = (antecedent, consequent)

    def __repr__(self) -> str:
        return f"({self.antecedent} -> {self.consequent})"


class Or(Prop):
    """A logical disjunction (P \\/ Q)."""

    def __init__(self, left: Prop, right: Prop):
        self.left = left
        self.right = right
        self.args = (left, right)

    def __repr__(self) -> str:
        return f"({self.left} \\/ {self.right})"


class PropFalse(Prop):
    """The proposition representing falsehood."""

    def __init__(self):
        self.args = ()

    def __repr__(self) -> str:
        return "False"

