from env.language import Prop

# Proof state abstractions can be any string
type AbstractState = str

# The value of a state is any float
type Value = float


class ProofState:
    """Represents the state of a proof, with goals and a context."""

    def __init__(self, goals: list[Prop], context: dict[str, Prop] | None = None):
        self.goals = goals
        self.context = context if context is not None else {}

    @property
    def is_complete(self) -> bool:
        """Checks if there are any goals left to prove."""
        return not self.goals

    @property
    def next_goal(self) -> Prop | None:
        """Returns the next goal to prove, or None if there are no goals."""
        return self.goals[0] if self.goals else None

    def __str__(self) -> str:
        if not self.context and not self.goals:
            return "Proof complete."

        ctx_str = "\n".join([f"{name}: {prop}" for name, prop in self.context.items()])
        goal_str = "\n".join([f"⊢ {g}" for g in self.goals])
        separator = "\n" + "=" * 20 + "\n"

        return f"{ctx_str}{separator}{goal_str}"

    def __hash__(self) -> int:
        # Hash ProofState by its string representation
        return self.__str__().__hash__()
