import argparse
import importlib
import random
from env.parser import parse_theorem_signature
from env.interaction import Trajectory


def main():
    parser = argparse.ArgumentParser(description="Run a proof search agent.")
    parser.add_argument(
        "--theorems",
        type=str,
        default="theorems.txt",
        help="File containing theorems to prove.",
    )
    parser.add_argument(
        "--max_attempts", type=int, required=True, help="Max rollouts per task."
    )
    parser.add_argument(
        "--agent",
        type=str,
        required=True,
        help="Which agent to use (e.g., part1, part2, part3, part4/a, part4/b).",
    )
    parser.add_argument(
        "--seed",
        type=int,
        required=False,
        default=None,
        help="Initial random seed for reproducibility.",
    )
    args = parser.parse_args()

    if args.seed is not None:
        random.seed(args.seed)

    # Dynamically import the agent
    agent_module_name = args.agent.replace("/", ".") + ".agent"
    try:
        agent_module = importlib.import_module(agent_module_name)
        agent = agent_module.Agent()
    except (ImportError, AttributeError) as e:
        print(f"Could not find or instantiate Agent in {agent_module_name}: {e}")
        return

    # Read and parse theorems
    with open(args.theorems, "r") as f:
        content = f.read().strip()

    # Better parsing: remove comments and extract theorems
    # Remove comment lines
    lines = content.split("\n")
    clean_lines = []
    for line in lines:
        if not line.strip().startswith("#"):
            clean_lines.append(line)
    clean_content = "\n".join(clean_lines)

    # Split theorems by double newline
    theorem_texts = clean_content.strip().split("\n\n")
    theorems = []
    for theorem_text in theorem_texts:
        theorem_text = theorem_text.strip()
        if theorem_text and theorem_text.startswith("theorem "):
            theorems.append(theorem_text)

    num_successes = 0
    total_theorems = 0
    for theorem_text in theorems:
        theorem_text = theorem_text.strip()
        if not theorem_text:
            continue

        theorem_sig = theorem_text.split(":=")[0].strip()
        print(f"Attempting to prove: {theorem_sig}")

        try:
            initial_state = parse_theorem_signature(theorem_sig)
        except Exception as e:
            print(f"Failed to parse theorem: {e}")
            continue

        total_theorems += 1
        proof: Trajectory | None = None
        num_successful_rollouts = 0
        for _ in range(args.max_attempts):
            # TODO: Your code for the main.py script goes here!
            # Each attempt should use the agent to perform a proof search.
            # The number of successful attempts should be counted in `num_successful_rollouts`.
            # If >= 1 successful rollout is found, populate `proof` with the trajectory of actions taken in that rollout.
            # (This is just to help you debug, and to see if the agent finds the same proofs as are listed in `theorems.txt`.)
            pass

        if proof is not None:
            num_successes += 1  # found at least one successful proof
            print(
                f"{num_successful_rollouts}/{args.max_attempts} successful rollouts for this theorem."
            )
            print("Example proof:")
            for _, action in proof:
                print(f"  {action}")

        else:
            print("Proof failed. No successful rollouts.")

        print("-" * 20)

    print(f"Total successful proofs: {num_successes}/{total_theorems}")


if __name__ == "__main__":
    main()
