(*
 * Copyright (c) 2009, Andrew Appel, Robert Dockins and Aquinas Hobor.
 *
 *)
(*
 * Modified 2010, Christian J. Bell; extended with history resources
 *
 *)

(** This library defines boolean algrbas defined from an order-theoretic
    perspective.  In short, a boolean algrbra is a complemented distributive
    lattice.  We additionally require that the boolean algrbra be non-trivial.

    From this definition we can recover the axioms of boolean algrbras as
    defined in universal algebra.

    We also define module interfaces for boolean algrbras with a splitting
    operator, with a relativization operator, and those which support
    token counting.

    We then say that a share model is a boolean algebra which satisfies
    all three module interfaces.  We also require that the elements
    have a decidable equality.


    Note: This file makes heavy use of math symbols in UTF8 encoding.
    You may need to adust your system to properly handle UTF8.
*)

Require Import base.
Require Import sepalg.

Module Type BOOLEAN_ALGEBRA.
  Parameters (t:Type) (Ord : t -> t -> Prop)
     (top bot : t) (lub glb : t -> t -> t) (comp : t -> t).

  Delimit Scope ba with ba. Open Scope ba.
  Notation "x 'uS' y" := (Ord x y) (at level 55, no associativity) : ba.
  Notation "'uT'" := top : ba.
  Notation "'uB'" := bot : ba.
  Notation "x 'uU' y" := (lub x y) (at level 45, right associativity) : ba.
  Notation "x 'uI' y" := (glb x y) (at level 47, right associativity) : ba.
  Notation "'uN' x" := (comp x) (at level 15, right associativity) : ba.

  Axiom ord_refl : forall x, x  uS  x.
  Axiom ord_trans : forall x y z, x  uS  y -> y  uS  z -> x  uS  z.
  Axiom ord_antisym : forall x y, x  uS  y -> y  uS  x -> x = y.

  Axiom lub_upper1 : forall x y, x  uS  x  uU  y.
  Axiom lub_upper2 : forall x y, y  uS  x  uU  y.
  Axiom lub_least : forall x y z, x  uS  z -> y  uS  z -> (x  uU  y)  uS  z.

  Axiom glb_lower1 : forall x y, (x  uI  y)  uS  x.
  Axiom glb_lower2 : forall x y, (x  uI  y)  uS  y.
  Axiom glb_greatest : forall x y z, z  uS  x -> z  uS  y -> z  uS  (x  uI  y).

  Axiom top_correct : forall x, x  uS   uT .
  Axiom bot_correct : forall x,  uB   uS  x.

  Axiom distrib1 : forall x y z, x  uI  (y  uU  z) = (x  uI  y)  uU  (x  uI  z).
  Axiom distrib2 : forall x y z, x  uU  (y  uI  z) = (x  uU  y)  uI  (x  uU  z).

  Axiom comp1 : forall x, x  uU   uN x =  uT .
  Axiom comp2 : forall x, x  uI   uN x =  uB .

  Axiom lat_nontrivial :  uT  <>  uB .

  Hint Resolve ord_refl ord_antisym lub_upper1 lub_upper2 lub_least
         glb_lower1 glb_lower2 glb_greatest top_correct bot_correct
         ord_trans distrib1 distrib2 : ba.
End BOOLEAN_ALGEBRA.

Module Type BA_FACTS.
  Declare Module BA:BOOLEAN_ALGEBRA.
  Export BA.

  Axiom ord_spec1 : forall x y, x  uS  y <-> x = x  uI  y.
  Axiom ord_spec2 : forall x y, x  uS  y <-> x  uU  y = y.

  Axiom lub_idem : forall x, (x  uU  x) = x.
  Axiom lub_commute : forall x y, x  uU  y = y  uU  x.
  Axiom lub_bot : forall x, x  uU   uB  = x.
  Axiom lub_top : forall x, x  uU   uT  =  uT .
  Axiom lub_absorb : forall x y, x  uU  (x  uI  y) = x.
  Axiom lub_assoc : forall x y z, (x  uU  y)  uU  z = x  uU  (y  uU  z).

  Axiom glb_idem : forall x, x  uI  x = x.
  Axiom glb_commute : forall x y, x  uI  y = y  uI  x.
  Axiom glb_bot : forall x, x  uI   uB  =  uB .
  Axiom glb_top : forall x, x  uI   uT  = x.
  Axiom glb_absorb : forall x y, x  uI  (x  uU  y) = x.
  Axiom glb_assoc : forall x y z, (x  uI  y)  uI  z = x  uI  (y  uI  z).

  Axiom distrib_spec : forall x y1 y2,
    x  uU  y1 = x  uU  y2 ->
    x  uI  y1 = x  uI  y2 ->
    y1 = y2.

  Axiom demorgan1 : forall x y,  uN (x  uU  y) =  uN x  uI   uN y.
  Axiom demorgan2 : forall x y,  uN (x  uI  y) =  uN x  uU   uN y.
  Axiom comp_inv : forall x,  uN  uN x = x.

  Definition ba_join (x y z:t) := x  uI  y =  uB  /\ x  uU  y = z.

  Axiom ba_saf: sepalgfacts ba_join.
  Definition ba_sa : sepalg t := SepAlgFromFacts ba_saf.
End BA_FACTS.

Module Type TOKEN_FACTORY.
  Declare Module BAF:BA_FACTS.
  Import BAF.

  Parameter isTokenFactory : t -> nat -> Prop.
  Parameter isToken : t -> nat -> Prop.

  Let join sa := @join t sa.

  Parameter splitToken : forall fac x n,
    isTokenFactory fac x ->
    { fac':t & {tok:t | isTokenFactory fac' (n+x) /\ isToken tok n /\ join ba_sa fac' tok fac}}.

  Axiom absorbToken : forall fac x n,
    { fac':t & {tok:t | isTokenFactory fac' (n+x) /\ isToken tok n /\ join ba_sa fac' tok fac}} ->
    isTokenFactory fac x.

  Axiom mergeToken : forall tok1 n1 tok2 n2 tok',
    isToken tok1 n1 ->
    isToken tok2 n2 ->
    join ba_sa tok1 tok2 tok' ->
    isToken tok' (n1+n2).

  Axiom unmergeToken : forall tok n1 n2,
    isToken tok (n1+n2) ->
    { tok1:t & { tok2:t &
      isToken tok1 n1 /\ isToken tok2 n2 /\
      join ba_sa tok1 tok2 tok }}.

  Axiom factoryOverlap : forall f1 f2 n1 n2,
    isTokenFactory f1 n1 -> isTokenFactory f2 n2 -> f1  uI  f2 <>  uB .

  Axiom fullFactory : forall x, isTokenFactory x 0 <-> x =  uT .
  Axiom identityToken : forall x, isToken x 0 <-> x =  uB .

  Axiom nonidentityToken : forall x n, (n > 0)%nat -> isToken x n -> x <>  uB .
  Axiom nonidentityFactory : forall x n, isTokenFactory x n -> x <>  uB .
End TOKEN_FACTORY.

Module Type SPLITTABLE_LATTICE.
  Declare Module BAF:BA_FACTS.
  Import BAF.

  Parameter split : t -> t * t.

  Axiom split_disjoint : forall x1 x2 x,
    split x = (x1, x2) ->
    x1  uI  x2 =  uB .

  Axiom split_together : forall x1 x2 x,
    split x = (x1, x2) ->
    x1  uU  x2 = x.

  Axiom split_nontrivial : forall x1 x2 x,
    split x = (x1, x2) ->
    (x1 =  uB  \/ x2 =  uB ) ->
    x =  uB .

End SPLITTABLE_LATTICE.

Module Type BA_RELATIV.
  Declare Module BAF:BA_FACTS.
  Import BAF.

  Parameter rel : t -> t -> t.

  Axiom rel_inj_l : forall a x y, a <>  uB  -> rel a x = rel a y -> x = y.
  Axiom rel_inj_r : forall a b x, x <>  uB  -> rel a x = rel b x -> a = b.

  Axiom rel_assoc : forall x y z, rel x (rel y z) = rel (rel x y) z.

  Axiom rel_preserves_glb : forall a x y, rel a (x  uI  y) = (rel a x)  uI  (rel a y).
  Axiom rel_preserves_lub : forall a x y, rel a (x  uU  y) = (rel a x)  uU  (rel a y).

  Axiom rel_bot1 : forall a, rel a  uB  =  uB .
  Axiom rel_bot2 : forall x, rel  uB  x =  uB .
  Axiom rel_top1 : forall a, rel a  uT  = a.
  Axiom rel_top2 : forall x, rel  uT  x = x.
End BA_RELATIV.

Module Type SHARE_MODEL.
  Declare Module BAF:BA_FACTS.

  Declare Module Splittable : SPLITTABLE_LATTICE
    with Module BAF:=BAF.
  Declare Module TokenFactory : TOKEN_FACTORY
    with Module BAF:=BAF.
  Declare Module Relativ : BA_RELATIV
    with Module BAF:=BAF.

  Export BAF.

  Parameter eq_dec : forall x y:t, {x=y} + {x<>y}.
End SHARE_MODEL.


Module BA_Facts (BA':BOOLEAN_ALGEBRA) :
  BA_FACTS with Module BA:=BA'.

  Module BA := BA'.
  Export BA.

  Lemma ord_spec1 : forall x y, x  uS  y <-> x = x  uI  y.
  Proof.
    split; intros.
    auto with ba.
    rewrite H; auto with ba.
  Qed.

  Lemma ord_spec2 : forall x y, x  uS  y <-> x  uU  y = y.
  Proof.
    intros; split; intros.
    auto with ba.
    rewrite <- H; auto with ba.
  Qed.

  Lemma lub_idem : forall x, x  uU  x = x.
  Proof. auto with ba. Qed.

  Lemma glb_idem : forall x, x  uI  x = x.
  Proof. auto with ba. Qed.

  Lemma lub_commute : forall x y, x  uU  y = y  uU  x.
  Proof. auto with ba. Qed.

  Lemma glb_commute : forall x y, x  uI  y = y  uI  x.
  Proof. auto with ba. Qed.

  Lemma lub_absorb : forall x y, x  uU  (x  uI  y) = x.
  Proof. auto with ba. Qed.

  Lemma glb_absorb : forall x y, x  uI  (x  uU  y) = x.
  Proof. auto with ba. Qed.

  Lemma lub_assoc : forall x y z, (x  uU  y)  uU  z = x  uU  (y  uU  z).
  Proof.
    intros; apply ord_antisym; eauto with ba.
  Qed.

  Lemma glb_assoc : forall x y z, (x  uI  y)  uI  z = x  uI  (y  uI  z).
  Proof.
    intros; apply ord_antisym; eauto with ba.
  Qed.

  Lemma glb_bot : forall x, x  uI   uB  =  uB .
  Proof. auto with ba. Qed.

  Lemma lub_top : forall x, x  uU   uT  =  uT .
  Proof. auto with ba. Qed.

  Lemma lub_bot : forall x, x  uU   uB  = x.
  Proof. auto with ba. Qed.

  Lemma glb_top : forall x, x  uI   uT  = x.
  Proof. auto with ba. Qed.

  Lemma distrib_spec : forall x y1 y2,
    x  uU  y1 = x  uU  y2 ->
    x  uI  y1 = x  uI  y2 ->
    y1 = y2.
  Proof.
    intros.
    rewrite <- (lub_absorb y2 x).
    rewrite glb_commute.
    rewrite <- H0.
    rewrite distrib2.
    rewrite lub_commute.
    rewrite <- H.
    rewrite (lub_commute x y1).
    rewrite (lub_commute y2 y1).
    rewrite <- distrib2.
    rewrite <- H0.
    rewrite glb_commute.
    rewrite lub_absorb.
    auto.
  Qed.

  Lemma comp_inv : forall x,  uN  uN x = x.
  Proof.
    intro x.
    apply distrib_spec with ( uN x).
    rewrite comp1.
    rewrite lub_commute.
    rewrite comp1.
    auto.
    rewrite comp2.
    rewrite glb_commute.
    rewrite comp2.
    auto.
  Qed.

  Lemma demorgan1 : forall x y,  uN (x  uU  y) =   uN x  uI   uN y.
  Proof.
    intros x y.
    apply distrib_spec with (x  uU  y).
    rewrite comp1.
    rewrite distrib2.
    rewrite (lub_assoc x y ( uN y)).
    rewrite comp1.
    rewrite lub_top.
    rewrite glb_top.
    rewrite (lub_commute x y).
    rewrite lub_assoc.
    rewrite comp1.
    rewrite lub_top.
    auto.
    rewrite comp2.
    rewrite glb_commute.
    rewrite distrib1.
    rewrite (glb_commute ( uN x) ( uN y)).
    rewrite glb_assoc.
    rewrite (glb_commute ( uN x) x).
    rewrite comp2.
    rewrite glb_bot.
    rewrite lub_commute.
    rewrite lub_bot.
    rewrite (glb_commute ( uN y) ( uN x)).
    rewrite glb_assoc.
    rewrite (glb_commute ( uN y) y).
    rewrite comp2.
    rewrite glb_bot.
    auto.
  Qed.

  Lemma demorgan2 : forall x y,  uN (x  uI  y) = ( uN x)  uU  ( uN y).
  Proof.
    intros x y.
    apply distrib_spec with (x  uI  y).
    rewrite comp1.
    rewrite lub_commute.
    rewrite distrib2.
    rewrite (lub_commute ( uN x) ( uN y)).
    rewrite lub_assoc.
    rewrite (lub_commute ( uN x) x).
    rewrite comp1.
    rewrite lub_top.
    rewrite glb_commute.
    rewrite glb_top.
    rewrite (lub_commute ( uN y) ( uN x)).
    rewrite lub_assoc.
    rewrite (lub_commute ( uN y) y).
    rewrite comp1.
    rewrite lub_top.
    auto.
    rewrite comp2.
    rewrite distrib1.
    rewrite (glb_commute x y).
    rewrite glb_assoc.
    rewrite comp2.
    rewrite glb_bot.
    rewrite lub_commute.
    rewrite lub_bot.
    rewrite (glb_commute y x).
    rewrite glb_assoc.
    rewrite comp2.
    rewrite glb_bot.
    auto.
  Qed.

  Definition ba_join (x y z:t) := x  uI  y =  uB  /\ x  uU  y = z.

  Lemma ba_saf: sepalgfacts ba_join.
  Proof.
    apply SepAlgFacts.

    (* saf_eq *)
    unfold ba_join; intuition; congruence.

    (* saf_assoc *)
    unfold ba_join; intuition.
    exists (b  uU  c); intuition.
    rewrite <- H2 in H.
    rewrite <- H.
    apply ord_antisym.
    eauto with ba.
    rewrite H; auto with ba.
    cut (a  uI  c =  uB ); intros.
    rewrite distrib1.
    rewrite H1.
    rewrite lub_commute; rewrite lub_bot; auto.
    apply ord_antisym.
    apply ord_trans with (d  uI  c).
    rewrite <- H2.
    eauto with ba.
    rewrite H; auto with ba.
    auto with ba.
    rewrite <- lub_assoc.
    rewrite H2; auto.

    (* saf_com *)
    unfold ba_join; intros a b c [H1 H2].
    rewrite glb_commute.
    rewrite lub_commute.
    auto.

    (* saf_cancellation *)
    unfold ba_join; intuition.
    apply distrib_spec with b.
    rewrite lub_commute; rewrite H2.
    rewrite lub_commute; rewrite H3.
    trivial.
    rewrite glb_commute; rewrite H1.
    rewrite glb_commute; rewrite H.
    trivial.

    (* saf_exist_units *)
    intros x; exists  uB ; split.
    rewrite glb_commute; apply glb_bot.
    rewrite lub_commute; apply lub_bot.

    (* saf_self_join *)
    intros.
    destruct H.
    rewrite lub_idem in H0.
    auto.
  Qed.

  Definition ba_sa : sepalg t :=  SepAlgFromFacts ba_saf.
End BA_Facts.
