(*
 * Copyright (c) 2009, Andrew Appel, Robert Dockins and Aquinas Hobor.
 *
 *)

(** This library builds a separation algrbra model of heaps as suggested
    in the paper.  We define the points-to operator in essentially
    the same way, and prove the claims about its behavior.
*)

Require Import base.
Require Import sepalg.
Require Import sepalg_generators.
Require Import shares.
Require Import sepalg_model.

Open Local Scope Z_scope.

(** We can build a heap model over an arbitrary set of locations [L] and
    values [V] provided that we have a decidable equality on [L].
*)
Module Type HEAP_SPEC.
  Parameter L:Set.
  Parameter V:Set.

  Parameter L_eq_dec : forall x y:L, {x=y}+{x<>y}.

End HEAP_SPEC.

Module HeapModel (HS:HEAP_SPEC).
  Import HS.

  (** Instantiate the system HBI using the heap model.
   *)
  Definition h :=
     L -> option (pjoinable (sa_prod V Share.t (sa_equiv V) Share.sa)).
  Definition h_sa : sepalg h :=
    sa_fun L _ (sa_lift (sa_prod _ _ (sa_equiv V) Share.sa)).

  Module HM_SA <: SEPALG.
    Definition A := h.
    Definition sa := h_sa.
  End HM_SA.

  Module HM_Soundness := Soundness(HM_SA).
  Import HM_Soundness.

  (** Now define the points-to operator.  This differs slightly from the paper,
      but embodies the same general idea.
   *)
  Definition points_to (l:L) (s:Share.t) (i:Z) (v:V) : formula :=
    fun h => exists p:Share.t, exists H,
      h l = Some (existT _ (v,Share.rel s p) H) /\
      (( i >= 0 /\ Share.isTokenFactory p (Zabs_nat i) ) \/
       ( i <  0 /\ Share.isToken p (Zabs_nat i))) /\
      (forall l', l <> l' -> h l' = None).

  (**  Points-to enjoys a generalization of Parkinson's "disjoint" axiom.
   *)
  Theorem points_to_disjoint : forall l v s i,
    star (points_to l s 0 v) (points_to l s i v) = bot.
  Proof.
    intros.
    extensionality x.
    unfold bot.
    apply prop_ext; intuition.
    destruct H as [h1 [h2 [? [? ?]]]].
    spec H l.
    destruct H0 as [p [Hp [? [? ?]]]].
    destruct H1 as [q [Hq [? [? ?]]]].
    simpl in H.
    destruct H; try discriminate.
    inv H0; inv H1.
    simpl in H.
    destruct z as [[v' s'] ?].
    destruct H; simpl in *.
    destruct H; subst v'.
    destruct H2.
    2: elimtype False; omega.
    destruct H1.
    rewrite Share.fullFactory in H2.
    subst p.
    rewrite Share.rel_top1 in Hp.
    rewrite Share.rel_top1 in H0.    
    generalize (Share.rel_leq s q).
    intros [w Hw].
    apply join_com in Hw.
    destruct (join_assoc _ _ _ _ _ _ Hw H0) as [c [? ?]].
    generalize (join_self _ _ _ H2); intros.
    subst c.
    assert (identity Share.sa s \/ identity Share.sa q).
    apply Share.rel_nontrivial.
    rewrite identity_unit_equiv; auto.
    destruct H7.
    apply Hp.
    red; intros.
    destruct a; destruct b; destruct H8; simpl in *.
    apply injective_projections; simpl.
    destruct H8; auto.
    apply H7; auto.
    intuition.
    apply Share.nonidentityFactory with q (Zabs_nat i); auto.
    apply Share.nonidentityToken with q (Zabs_nat i); auto.
    red.
    apply inj_lt_rev; simpl.
    rewrite inj_Zabs_nat.
    rewrite Zabs_non_eq; omega.
  Qed.

  (** In any heap with a tokenized points-to fact, we can find a share [s'] which
      is the bare, untokenized share.
    *)
  Theorem points_to_swap : forall l s i v h,
    points_to l s i v h -> exists! s', points_to l s i v h = points_to l s' 0 v h.
  Proof.
    intros.
    generalize H; intro H0.
    destruct H as [p [Hp [? [? ?]]]].
    exists (Share.rel s p).
    split.
    apply prop_ext; split; auto.
    intros _.
    red.
    exists Share.top.
    rewrite Share.rel_top1.
    exists Hp.
    split; auto.
    split; auto.
    left; split; auto.
    omega.
    simpl.
    rewrite Share.fullFactory; auto.

    intros.
    rewrite H3 in H0.
    destruct H0 as [p' [? [? ?]]].
    rewrite H in H0; inv H0.
    assert (p' = Share.top).
    destruct H4.
    destruct H0.
    destruct H0.
    simpl in H5.
    rewrite Share.fullFactory in H5; auto.
    destruct H0; elimtype False; omega.
    rewrite H6.
    rewrite H0.
    rewrite Share.rel_top1.
    auto.
  Qed.

  (** This defines the join relation of the "counting" share model,
      leaving out the unit.  *)
  Inductive count_join : Z -> Z -> Z -> Prop :=
    | cnt_join : forall x y,
         (x < 0 \/ y < 0) ->
         (x + y >= 0) \/ (x < 0 /\ y < 0) ->
         count_join x y (x+y).

  (** Prove that points-to satisfies the token counting law. *)
  Theorem points_to_token : forall l s v i1 i2 i3,
    count_join i1 i2 i3 ->
    (star (points_to l s i1 v) (points_to l s i2 v) = points_to l s i3 v).
  Proof.
    intros l s v i1 i2 i3 Hi.
    extensionality x.
    apply prop_ext; intuition.

    (* -> direction *)
    destruct H as [h1 [h2 [? [? ?]]]].
    destruct H0 as [p [Hp [? [? ?]]]].
    destruct H1 as [q [Hq [? [? ?]]]].
    generalize H; intro Hh.
    spec H l.
    red.
    inv H.
    rewrite <- H7 in H0; discriminate.
    rewrite <- H8 in H1; discriminate.
    rewrite <- H6 in H0.
    rewrite <- H7 in H1.
    clear H6 H7 H8.
    inv H0; inv H1.
    simpl in H9.
    destruct z as [[v' s'] ?].
    destruct H9; simpl in *.
    destruct H; subst v'.
    inv Hi.
    destruct (Share.rel_join2 s p q s').
    assert (nonidentity Share.sa (Share.rel s p)).

    hnf; intros.
    apply Hp.
    red; intros.
    destruct a; destruct b; destruct H8; simpl in *.
    destruct H8; subst v1.
    replace t0 with t; auto.
    red; intro; apply H7.
    apply identity_share_bot in H8.
    subst s.
    rewrite Share.rel_bot2; auto.
    apply Share.bot_identity.
    auto.
    destruct H7.
    subst s'.
    exists x0.
    exists n.
    split; auto.
    split.

    intuition (try (elimtype False; omega)).
    left; split; auto.
    apply Share.absorbToken with (Zabs_nat i2).
    exists p.
    exists q.
    intuition.
    replace (Zabs_nat i2 + (Zabs_nat (i1 + i2)))%nat with (Zabs_nat i1); auto.
    rewrite inj_eq_iff.
    rewrite inj_Zabs_nat.
    rewrite inj_plus.
    rewrite inj_Zabs_nat.
    rewrite (Zabs_non_eq i2); try omega.
    rewrite inj_Zabs_nat.    
    rewrite Zabs_eq; try omega.
    rewrite Zabs_eq; try omega.
    left; split; auto.
    apply Share.absorbToken with (Zabs_nat i1).
    exists q; exists p.
    intuition.
    replace (Zabs_nat i1 + (Zabs_nat (i1 + i2)))%nat with (Zabs_nat i2); auto.
    rewrite inj_eq_iff.
    rewrite inj_Zabs_nat.
    rewrite inj_plus.
    rewrite inj_Zabs_nat.
    rewrite (Zabs_non_eq i1); try omega.
    rewrite inj_Zabs_nat.    
    rewrite Zabs_eq; try omega.
    rewrite Zabs_eq; try omega.    
    apply join_com; auto.

    right; split; try omega.
    replace (Zabs_nat (i1 + i2)) with (Zabs_nat i1 + Zabs_nat i2)%nat.
    eapply Share.mergeToken; eauto.
    rewrite inj_eq_iff.
    rewrite inj_Zabs_nat.
    rewrite inj_plus.
    repeat rewrite inj_Zabs_nat.
    rewrite (Zabs_non_eq i1); try omega.
    rewrite (Zabs_non_eq i2); try omega.
    rewrite (Zabs_non_eq (i1+i2)); try omega.
    
    right; split; try omega.
    replace (Zabs_nat (i1 + i2)) with (Zabs_nat i1 + Zabs_nat i2)%nat.
    eapply Share.mergeToken; eauto.
    rewrite inj_eq_iff.
    rewrite inj_Zabs_nat.
    rewrite inj_plus.
    repeat rewrite inj_Zabs_nat.
    rewrite (Zabs_non_eq i1); try omega.
    rewrite (Zabs_non_eq i2); try omega.
    rewrite (Zabs_non_eq (i1+i2)); try omega.

    intros.
    spec Hh l'.
    inv Hh; auto.
    spec H5 l' H7.
    rewrite <- H10 in H5; discriminate.

    (* <- direction *)

    destruct H as [s' [H [? [? ?]]]].

    inv Hi.
    cut (exists p, exists q, join Share.sa p q s' /\
      ((i1 >= 0) /\ Share.isTokenFactory p (Zabs_nat i1) \/
       (i1 <  0) /\ Share.isToken p (Zabs_nat i1)) /\
      ((i2 >= 0) /\ Share.isTokenFactory q (Zabs_nat i2) \/
       (i2 <  0) /\ Share.isToken q (Zabs_nat i2))).
    intros [p [q [? [? ?]]]].
    cut (nonidentity (sa_prod _ _ (sa_equiv V) Share.sa) (v,Share.rel s p)).
    intro Hnonid1.
    cut (nonidentity (sa_prod _ _ (sa_equiv V) Share.sa) (v,Share.rel s q)).
    intro Hnonid2.

    set (h1 := fun l':L => if (L_eq_dec l l') then Some (existT _ (v,Share.rel s p) Hnonid1) else None).
    set (h2 := fun l':L => if (L_eq_dec l l') then Some (existT _ (v,Share.rel s q) Hnonid2) else None).
    exists h1; exists h2.
    split.
    simpl; intro l'.
    unfold h1,h2.
    destruct (L_eq_dec l l'); simpl.
    subst l'.
    rewrite H0.
    simpl.
    constructor.
    constructor.
    simpl; split; auto.
    simpl.
    apply Share.rel_join; auto.
    rewrite H2; auto.
    constructor.
    split.
    red.
    exists p; exists Hnonid1.
    split.
    unfold h1.
    destruct (L_eq_dec l l).
    auto.
    elim n; auto.
    split; auto.
    intros.
    unfold h1.
    destruct (L_eq_dec l l'); auto.
    subst l'.
    elim H8; auto.
    red.
    exists q; exists Hnonid2.
    split.
    unfold h2.
    destruct (L_eq_dec l l); simpl; auto.
    elim n; auto.
    split; auto.
    intros.
    unfold h2.
    destruct (L_eq_dec l l'); simpl.
    subst l'.
    elim H8; auto.
    auto.

    intro.
    assert (identity Share.sa (Share.rel s q)).
    red; intros.
    assert ((v,a) = (v,b)).
    apply H8.
    split; simpl; auto.
    split; auto.
    inv H10; auto.
    destruct (Share.rel_nontrivial _ _ H9).
    apply H.
    apply identity_share_bot in H10.
    subst s.
    rewrite Share.rel_bot2.
    red; intros.
    destruct a; destruct b; destruct H10; simpl in *.
    destruct H10; subst.
    replace t0 with t; auto.
    apply Share.bot_identity; auto.
    do 2 destruct H7.
    apply (Share.nonidentityFactory q (Zabs_nat i2)); auto.
    apply (Share.nonidentityToken q (Zabs_nat i2)); auto.
    red.
    apply inj_lt_rev.
    rewrite inj_Zabs_nat.
    rewrite Zabs_non_eq.
    simpl.
    omega.
    omega.

    intro.
    assert (identity Share.sa (Share.rel s p)).
    red; intros.
    assert ((v,a) = (v,b)).
    apply H8.
    split; simpl; auto.
    split; auto.
    inv H10; auto.
    destruct (Share.rel_nontrivial _ _ H9).
    apply H.
    apply identity_share_bot in H10.
    subst s.
    rewrite Share.rel_bot2.
    red; intros.
    destruct a; destruct b; destruct H10; simpl in *.
    destruct H10; subst.
    replace t0 with t; auto.
    apply Share.bot_identity; auto.
    do 2 destruct H6.
    apply (Share.nonidentityFactory p (Zabs_nat i1)); auto.
    apply (Share.nonidentityToken p (Zabs_nat i1)); auto.
    red.
    apply inj_lt_rev.
    rewrite inj_Zabs_nat.
    rewrite Zabs_non_eq.
    simpl.
    omega.
    omega.

    intuition (try (elimtype False; omega)).
    
    generalize (Share.splitToken s' (Zabs_nat (i1+i2)) (Zabs_nat i1) H6).
    intros [q [p [? [? ?]]]].
    exists p; exists q; split; auto.
    apply join_com; auto.
    split.
    right; split; auto.
    left; split; try omega.
    replace (Zabs_nat i2) with (Zabs_nat i1 + Zabs_nat (i1 + i2))%nat; auto.
    
    rewrite inj_eq_iff.
    rewrite inj_Zabs_nat.
    rewrite inj_plus.
    repeat rewrite inj_Zabs_nat.
    rewrite (Zabs_non_eq i1); try omega.
    rewrite Zabs_eq; try omega.
    rewrite Zabs_eq; try omega.
    replace (Zabs_nat (i1 + i2)) with (Zabs_nat i1 + Zabs_nat i2)%nat in H7.
    generalize (Share.unmergeToken s' (Zabs_nat i1) (Zabs_nat i2) H7).
    intros [p [q [? [? ?]]]].
    exists p; exists q; split; auto.
    rewrite inj_eq_iff.
    rewrite inj_Zabs_nat.
    rewrite inj_plus.
    repeat rewrite inj_Zabs_nat.
    rewrite (Zabs_non_eq i1); try omega.
    rewrite (Zabs_non_eq i2); try omega.
    rewrite (Zabs_non_eq (i1+i2)); try omega.

    generalize (Share.splitToken s' (Zabs_nat (i1 + i2)) (Zabs_nat i2) H6).
    intros [p [q [? [? ?]]]].
    exists p; exists q; split; auto.
    split.
    left; split; try omega.
    replace (Zabs_nat i1) with (Zabs_nat i2 + Zabs_nat (i1 + i2))%nat; auto.
    rewrite inj_eq_iff.
    rewrite inj_Zabs_nat.
    rewrite inj_plus.
    repeat rewrite inj_Zabs_nat.
    rewrite (Zabs_non_eq i2); try omega.
    rewrite (Zabs_eq (i1+i2)); try omega.    
    rewrite (Zabs_eq i1); try omega.
    right; split; auto.
    
    replace (Zabs_nat (i1 + i2)) with (Zabs_nat i1 + Zabs_nat i2)%nat in H7.
    generalize (Share.unmergeToken s' (Zabs_nat i1) (Zabs_nat i2) H7).
    intros [p [q [? [? ?]]]].
    exists p; exists q; split; auto.
    rewrite inj_eq_iff.
    rewrite inj_Zabs_nat.
    rewrite inj_plus.
    repeat rewrite inj_Zabs_nat.
    rewrite (Zabs_non_eq i1); try omega.
    rewrite (Zabs_non_eq i2); try omega.
    rewrite (Zabs_non_eq (i1+i2)); try omega.    
  Qed.

  (** Prove that points-to satisfies the splitting law. *)
  Theorem points_to_split : forall l v (s1 s2 s3:pshare),
    join Share.sa s1 s2 s3 ->
       (star (points_to l s1 0 v) (points_to l s2 0 v) = points_to l s3 0 v).
  Proof.
    intros.
    extensionality x.
    apply prop_ext; split; intros.
    destruct H0 as [h1 [h2 [? [? ?]]]].
    destruct H1 as [p [Hp [? ?]]].
    destruct H2 as [q [Hq [? ?]]].
    generalize H0; intro Hh.
    spec H0 l.
    simpl in H0.
    red.
    destruct H0; try discriminate.
    inv H1; inv H2.
    destruct z as [[v' s] Hz].
    simpl in H0.
    destruct H0; simpl in *.
    destruct H0.
    subst v'.
    clear H0.
    intuition; try (elimtype False; omega).
    rewrite Share.fullFactory in H6.
    rewrite Share.fullFactory in H7.
    subst p q.
    repeat rewrite Share.rel_top1 in H1.
    exists Share.top.
    match goal with [ |- @ex ((identity ?Y ?X) -> False) _ ] =>
      assert (Hnid : nonidentity Y X) ; [ | exists Hnid ]
    end.
    apply join_nonidentity with (v,pshare_sh s1) (v,pshare_sh s2).
    rewrite Share.rel_top1 in Hp.
    rewrite Share.rel_top1 in Hq.
    auto.
    split; simpl; auto.
    split; auto.
    rewrite Share.rel_top1.
    auto.
    split.
    revert Hnid.
    rewrite Share.rel_top1.
    replace (pshare_sh s3) with s.
    intro Hnid.
    replace Hz with Hnid by apply proof_irr; auto.
    eapply join_eq; eauto.
    split.
    left; split; auto.
    rewrite Share.fullFactory; auto.
    intros.
    spec Hh l'.
    spec H2 l'.
    spec H5 l'.
    destruct Hh.
    apply H5; auto.
    apply H2; auto.
    apply H2 in H4; discriminate.

    destruct H0 as [s [Hs [? [? ?]]]].
    intuition; try (elimtype False; omega).
    simpl in *.
    rewrite Share.fullFactory in H4.
    subst s.
    revert H0.
    revert Hs.
    rewrite Share.rel_top1.
    intros.
    cut (nonidentity (sa_prod V Share.t (sa_equiv V) Share.sa) (v,pshare_sh s1)).
    intro Hnonid1.
    cut (nonidentity (sa_prod V Share.t (sa_equiv V) Share.sa) (v,pshare_sh s2)).
    intro Hnonid2.
    set (h1 := fun l' => if L_eq_dec l l' then Some (existT _ (v,pshare_sh s1) Hnonid1) else None).
    set (h2 := fun l' => if L_eq_dec l l' then Some (existT _ (v,pshare_sh s2) Hnonid2) else None).
    exists h1; exists h2.
    split.
    simpl; red; intro l'.
    unfold h1, h2.
    destruct (L_eq_dec l l').
    subst l'.
    rewrite H0.
    constructor.
    constructor.
    simpl; split; auto.
    simpl; auto.
    rewrite H2; auto.
    constructor.

    split.
    exists Share.top.
    rewrite Share.rel_top1.
    exists Hnonid1.
    split.
    unfold h1.
    destruct (L_eq_dec l l); auto.
    elim n; auto.
    split.
    left; split; auto.
    simpl.
    rewrite Share.fullFactory; auto.
    intros.
    unfold h1.
    destruct (L_eq_dec l l'); auto.
    subst l'.
    elim H3; auto.
    exists Share.top.
    rewrite Share.rel_top1.
    exists Hnonid2.
    split.
    unfold h2.
    destruct (L_eq_dec l l); auto.
    elim n; auto.
    split.
    left; split; auto.
    simpl.
    rewrite Share.fullFactory; auto.
    intros.
    unfold h2.
    destruct (L_eq_dec l l'); auto.
    subst l'; elim H3; auto.
    
    intro.
    rewrite identity_unit_equiv in H3.
    destruct H3; simpl in *.
    destruct s2; simpl in *.
    apply n.
    rewrite identity_unit_equiv; auto.

    intro.
    rewrite identity_unit_equiv in H3.
    destruct H3; simpl in *.
    destruct s1; simpl in *.
    apply n.
    rewrite identity_unit_equiv; auto.
  Qed.
End HeapModel.
