(*
 * Copyright 2010, Christian J. Bell
 *
 *)

Require Import List.
Require Import Morphisms.
Require Import SetoidClass.
Require Import SetoidTactics.
Require Import sepalg.

Module Type HistSet_Value_SPEC.
  Parameter V: Type.
End HistSet_Value_SPEC.

Module Type HistSet_SPEC (HSV: HistSet_Value_SPEC).
  Definition V:= HSV.V.
  Parameter HistSet : Type.
  Definition Hist := list V.

  Parameter In : Hist -> HistSet -> Prop.
  Parameter Merge: HistSet -> HistSet -> HistSet.
  Parameter hCons : V -> HistSet -> HistSet.
  Parameter Filter : forall P H (P_dec: forall h, {P h}+{~P h}), {H' : HistSet & forall h, In h H' <-> (In h H /\ P h) }.
  Parameter Exists : forall P H (P_dec: forall h, {P h}+{~P h}), {exists h, In h H /\ P h}+{~exists h, In h H /\ P h}.
  Parameter histset_unit : HistSet.


  Definition Equiv H H' := forall h, In h H <-> In h H'.

  Program Instance EquivSetoid : Setoid HistSet :=
    { equiv:= Equiv }.
  Next Obligation.
    firstorder.
  Qed.

  Axiom In_dec: forall h H, {In h H}+{~In h H}.
  Axiom V_eq_dec: forall (x y:V), {x=y}+{x<>y}.

  Axiom merge_morphism: forall x x' y y', x==x' -> y==y' -> Merge x y == Merge x' y'.

  Axiom merge_associative: forall a b c, Merge (Merge a b) c == Merge a (Merge b c).
  Axiom merge_commutative: forall a b, Merge a b == Merge b a.
  Axiom merge_cancellative: forall a1 a2 b, Merge a1 b == Merge a2 b -> a1 == a2.
  Definition merge_unit_for : HistSet -> HistSet -> Prop := fun e a => Merge e a == a.
  Axiom merge_exist_units: forall a, {e : HistSet & merge_unit_for e a}.

  Axiom hCons_In: forall v h H, In h H <-> In (v::h) (hCons v H).
  Axiom hCons_v_eq: forall  v v' h H, In (v :: h) (hCons v' H) -> v=v'.
  Axiom histset_nonempty : forall H, {h : Hist & In h H}.
  Axiom histset_permutations : forall H h h', In h H -> In h' H -> Permutation h h'.
  Axiom histset_unit_merge: forall a, Merge histset_unit a == a.

End HistSet_SPEC.


Module HistSet_Morphisms (HSV: HistSet_Value_SPEC) (HS : HistSet_SPEC HSV).
  Import HS.

  Lemma histset_nil_hCons: forall H v, ~In nil (hCons v H).
  Proof.
    intros. intro.
    destruct (histset_nonempty H) as [h H1].
    apply (hCons_In v) in H1.
    eapply Permutation_nil_cons.
    apply (histset_permutations _ _ _ H0 H1).
  Qed.

  Add Morphism In with signature eq ==> Equiv ==> iff as In_m.
    firstorder.
  Qed.  

  Lemma hCons_morphism: forall v H H', H == H' -> hCons v H == hCons v H'.
    intros.
    split; intros.

    destruct h.
    elimtype False.
    eapply histset_nil_hCons.
    apply H1.
    substitute (hCons_v_eq _ _ _ _ H1).
    apply (hCons_In v h _).
    eapply hCons_In in H1.
    substitute H0.
    trivial.

    destruct h.
    elimtype False.
    eapply histset_nil_hCons.
    apply H1.
    substitute (hCons_v_eq _ _ _ _ H1).
    apply (hCons_In v h _).
    eapply hCons_In in H1.
    substitute H0.
    trivial.
  Qed.
    
  Add Morphism Merge : Merge_m.
    intros.
    apply merge_morphism; auto.
  Qed.  

  Add Morphism hCons with signature eq ==> Equiv ==> Equiv as hCons_m.
    intros.
    apply hCons_morphism.
    auto.
  Qed.

  Lemma Filter_Equiv: forall P1 P2 H1 H2 P1_dec P2_dec,
    (forall h, P1 h <-> P2 h) -> H1==H2 -> projT1 (Filter P1 H1 P1_dec) == projT1 (Filter P2 H2 P2_dec).
  Proof.
    intros.
    destruct (Filter P1 H1 P1_dec).
    destruct (Filter P2 H2 P2_dec).
    simpl.
    
    split; intros.
    specialize (i h); specialize (i0 h); firstorder.
    specialize (i h); specialize (i0 h); firstorder.
  Qed.

(*  Lemma Exists_Equiv (P:Hist->Prop) (H1 H2:HistSet) (P_dec: forall h, {P h}+{~P h}): H1==H2->{exists h, In h H2 /\ P h}+{~exists h, In h H2 /\ P h}.
    intros.
    destruct (Exists P H1 P_dec).
    left.
    substitute H.
    apply e.
    right.
    substitute H.
    apply n.
  Defined.
Check Exists_Equiv.

  Add Morphism (fun H P Pd => Exists P H Pd) with signature Equiv ==> sumbool _ _ as Exists_m.
    intros.
    apply hCons_morphism.
    auto.
  Qed.

*)

End HistSet_Morphisms.


Module HistSet_WeakSepAlg (HSV: HistSet_Value_SPEC) (HS : HistSet_SPEC HSV).
  Module HSM := HistSet_Morphisms HSV HS.
  Import HS.
  Import HSM.
  Require Import SetoidAxioms.

  Definition merge x y z:= Merge x y == z.

  Program Instance wsa_histset: wsepalg HistSet := {
    join:= merge; wsa_unit_for := fun e a => merge e a a
  }.
  Next Obligation (* functional *).
    apply setoideq_eq; firstorder.
  Qed.
  Next Obligation (* associative *).
    unfold merge in *.
    exists (Merge b c).
    split.
    apply setoid_refl.
    apply setoid_trans with (Merge (Merge a b) c).
    apply setoid_sym.
    apply merge_associative.
    substitute H; auto.
  Qed.
  Next Obligation (* commutative *).
    unfold merge in *.
    substitute<- H.
    apply merge_commutative.
  Qed.
  Next Obligation (* cancellative *).
    apply setoideq_eq.
    unfold merge in *.
    apply merge_cancellative with b.
    substitute H.
    substitute H0.
    apply setoid_refl.
  Qed.
  Next Obligation (* exist_units *).
    apply merge_exist_units. 
  Qed.

  Lemma histset_unit_identity: identity wsa_histset histset_unit.
  Proof.
    unfold identity; intros.
    unfold wsa_histset in H.
    simpl in H.
    unfold merge in H.
    apply setoideq_eq.
    setoid_rewrite<- (histset_unit_merge a).
    auto.
  Qed.   

  Lemma histset_unit_unique: forall a, identity wsa_histset a -> a == histset_unit.
  Admitted.



End HistSet_WeakSepAlg.


Module HistSet_Intersection (HSV: HistSet_Value_SPEC) (HS : HistSet_SPEC HSV).
  Module HSM := HistSet_Morphisms HSV HS.
  Import HS.
  Import HSM.

  Definition MatchTail (h ht : Hist) := exists l, h=l++ht.
  Lemma MatchTail_dec : forall (h ht : Hist), {MatchTail h ht}+{~MatchTail h ht}.
    intros.
    unfold MatchTail.
    assert ({(exists l : list V, rev h = rev (l ++ ht))} + {~ (exists l : list V, rev h = rev (l ++ ht))}).
    induction h; intros.
    admit.
    destruct (IHh).
    left.
    destruct e as [l H].
    exists (a::l).
    simpl.
    rewrite H.
    auto.

    destruct ht.
    elimtype False.
    apply n. exists h. rewrite distr_rev. auto.
    destruct (list_eq_dec V_eq_dec (a::h) (v::ht)).
    rewrite e.
    left.
    exists nil.
    auto.
    right.
    intro.
    destruct H.

    destruct x.

    simpl in H.
    apply app_inj_tail in H.
    destruct H.
    subst.
    apply n0.
    replace h with ht; auto.
    replace ht with (rev (rev ht)); auto.
    replace h with (rev (rev h)); auto.
    rewrite H.
    auto.
    apply rev_involutive.
    apply rev_involutive.

    assert (h = x++v::ht).
    simpl in H.
    apply app_inj_tail in H.
    destruct H.
    rewrite<- rev_involutive.
    rewrite<- H.
    rewrite rev_involutive.
    auto.
    apply n.
    exists x.
    rewrite H0; auto.

    destruct H.
    left.
    destruct e as [l H].
    exists l.
    rewrite<- rev_involutive.
    rewrite<- H.
    rewrite rev_involutive.
    auto.

    right.
    intro; apply n; clear n.
    destruct H as [l H].
    exists l.
    subst.
    auto.
  Qed.    

  Definition interL Hp Hc := 
    (Filter (fun hp : Hist => exists hc : Hist, In hc Hc /\ MatchTail hp hc) Hp
    (fun hp => Exists (fun hc : Hist => MatchTail hp hc) Hc (MatchTail_dec hp))
    ) : {Hp' : HistSet & forall hp, In hp Hp' <-> (In hp Hp /\ exists hc, In hc Hc /\ MatchTail hp hc)}.

  Definition interS Hp Hc := 
    (Filter (fun hc : Hist => exists hp : Hist, In hp Hp /\ MatchTail hp hc) Hc
    (fun hc => Exists (fun hp : Hist => MatchTail hp hc) Hp (fun hp => MatchTail_dec hp hc))
    ) : {Hc' : HistSet & forall hc, In hc Hc' <-> (In hc Hc /\ exists hp, In hp Hp /\ MatchTail hp hc)}.

  Definition intersect H1 H2:= Filter (fun h : Hist => In h H2) H1 (fun h : Hist => In_dec h H2)
     : {H : HistSet &  forall h : Hist, In h H <-> In h H1 /\ In h H2}.

  Lemma MatchTail_eq_length: forall h1 h2, length h1 = length h2 -> MatchTail h1 h2 -> h1 = h2.
  Proof.
    intros.
    destruct H0.
    destruct x.
    auto.

    assert (forall a b, S a + b <> b).
      clear.
      intros. intro.
      induction b.
      inversion H.
      inversion H.
      apply IHb.
      rewrite<- H1 at 2.
      simpl; auto.

    elimtype False.
    rewrite H0 in H.
    rewrite app_length in H.
    apply (H1 _ _ H).
  Qed.

  Definition histset_length_eq H1 H2:= forall h1 h2, In h1 H1 -> In h2 H2 -> length h1 = length h2.

  Lemma interL_eq_length: forall H1 H2, histset_length_eq H1 H2 -> projT1 (interL H1 H2) == projT1 (intersect H1 H2).
  Proof.
    intros.
    destruct (interL H1 H2).
    destruct (intersect H1 H2).
    simpl.
    split; intros.

    specialize (i h).
    destruct i.
    apply H3 in H0.
    destruct H0.
    eapply i0.
    split; auto.
    destruct H5.
    destruct H5.
    replace h with x1; auto.
    symmetry.
    apply MatchTail_eq_length; auto.

    specialize (i h).
    destruct i.
    apply H4.
    split.
    eapply i0; auto.
    exists h.
    split.
    eapply i0; auto.
    exists nil.
    auto.
  Qed.
    
  Lemma interS_eq_length: forall H1 H2, histset_length_eq H1 H2 -> projT1 (interS H1 H2) == projT1 (intersect H1 H2).
  Proof.
    intros.
    destruct (interS H1 H2).
    destruct (intersect H1 H2).
    simpl.
    split; intros.

    specialize (i h).
    destruct i.
    apply H3 in H0.
    destruct H0.
    eapply i0.
    split; auto.
    destruct H5.
    destruct H5.
    replace h with x1; auto.
    apply MatchTail_eq_length; auto.

    specialize (i h).
    destruct i.
    apply H4.
    split.
    eapply i0; auto.
    exists h.
    split.
    eapply i0; auto.
    exists nil.
    auto.
  Qed.

  Lemma interLS_eq_length: forall H1 H2, histset_length_eq H1 H2 -> projT1 (interL H1 H2) == projT1 (interS H1 H2).
  Proof.
    intros.
    apply setoid_trans with (projT1 (intersect H1 H2)).
    apply interL_eq_length; auto.
    apply setoid_sym.
    apply interS_eq_length; auto.
  Qed.

  Add Morphism (fun Hp Hc => projT1 (interL Hp Hc)) : interL_m.
    intros.
    apply Filter_Equiv; auto.
    substitute H0.
    intros; split; auto.
  Qed.

  Add Morphism (fun Hp Hc => projT1 (interS Hp Hc)) : interS_m.
    intros.
    apply Filter_Equiv; auto.
    substitute H.
    intros; split; auto.
  Qed.

  Add Morphism (fun H1 H2 => projT1 (intersect H1 H2)) : intersect_m.
    intros.
    apply Filter_Equiv; auto.
  Qed.

End HistSet_Intersection.






