(*
 * Copyright (c) 2009, Andrew Appel, Robert Dockins and Aquinas Hobor.
 *
 *)
(*
 * Modified 2010, Christian J. Bell; extended with history resources
 *
 *)

Require Import base.
Require Import sepalg.
Require Import sepalg_generators.

(** Proof of soundness of the induced separation logic
    with respect to the Hilbert-style system HBI.
 
    Table 2, page 265 in,

    Pym, "Possible worlds and resources: the semantics of BI",
       Theoretical Computer Science, 315 (2004) 257-305.
*)

Module Type SEPALG.
  Variable A:Type.
  Variable sa:sepalg A.  
End SEPALG.


Module Soundness (SA:SEPALG).
  Import SA.

  (** Interpretaions of the operators in the system *)
  Definition formula := A -> Prop.
  Definition entails (x y:formula) := forall a:A, x a -> y a.

  Definition top := fun a:A => True.
  Definition bot := fun a:A => False.
  Definition and (x y:formula) := fun a:A => x a /\ y a.
  Definition or  (x y:formula) := fun a:A => x a \/ y a.
  Definition impl (x y:formula) := fun a:A => x a -> y a.

  Let join sa := @join A sa.

  Definition emp := fun a:A, join sa a a a.
  Definition star (x y:formula) := fun a:A,
    exists a1, exists a2, join sa a1 a2 a /\ x a1 /\ y a2.
  Definition wand (x y:formula) := fun a:A,
    forall a1 a', join sa a1 a a' -> x a1 -> y a'.


  (** Soundness proofs for the axioms *)

  Module Notation.
    Delimit Scope predicate_scope with predicate.
    Delimit Scope forcing_scope with forcing.
    Definition forces (w:A) (P:formula) := P w.
    Notation "w |= P" := (forces w P%predicate) (no associativity, at level 70) : forcing_scope.
    Delimit Scope entails_scope with entails.
    Notation "P |- Q" := (entails (P%predicate) (Q%predicate)) (no associativity, at level 60) : entails_scope.
    Hint Unfold forces.

    Delimit Scope predicate_scope with predicate.
(*    Notation "P == Q" := (equ (P%predicate) (Q%predicate)) (no associativity, at level 70) : predicate_scope.*)
    Notation "P => Q" := (impl (P%predicate) (Q%predicate)) (right associativity, at level 55) : predicate_scope.
    Notation "P || Q" := (or (P%predicate) (Q%predicate)) : predicate_scope.
    Notation "P && Q" := (and (P%predicate) (Q%predicate)) : predicate_scope.
    Notation "P * Q" := (star (P%predicate) (Q%predicate)) (left associativity, at level 40) : predicate_scope.
    Notation "P =* Q" := (wand (P%predicate) (Q%predicate)) (right associativity, at level 55) : predicate_scope.
  End Notation.

  Section Facts.
    Import Notation.
    Open Scope forcing_scope.
    Open Scope entails_scope.

  Definition equ (x y:formula) := x |- y /\ y |- x.

  Lemma refl : forall x, x |- x.
  Proof.
    compute; auto.
  Qed.

  Lemma entails_trans: forall B A C, A |- B -> B |- C -> A |-  C.
  Proof.
    firstorder.
  Qed.

  Lemma entails_forces: forall A w B, w|=A -> A|-B -> w|=B.
    eauto.
  Qed.



  Lemma top_intro : forall x, x |- top.
  Proof.
    compute; auto.
  Qed.

  Lemma bot_elim : forall x, bot |- x.
  Proof.
    compute; intuition.
  Qed.

  Lemma and_intro : forall x y z,
    x |- y ->
    x |- z ->
    x |- y && z.
  Proof.
    unfold entails, and; intuition.
  Qed.

  Lemma and_elim1 : forall x y z,
    x |- y && z -> x |- y.
  Proof.
    unfold entails, and; intuition.
    destruct (H a); auto.
  Qed.

  Lemma and_elim3: forall A B C, A |- C -> A&&B |- C.
    intros. apply entails_trans with A0; auto. intro; intros. eapply H0. Qed.
  Lemma and_elim4: forall A B C, B |- C -> A&&B |- C.
    intros. apply entails_trans with B; auto. intro; intros. eapply H0. Qed.

  Lemma and_elim2 : forall x y z,
    x |- y && z -> x |- z.
  Proof.
    unfold entails, and; intuition.
    destruct (H a); auto.
  Qed.

  Lemma or_elim : forall x y z,
    x |- z ->
    y |- z ->
    x || y |- z.
  Proof.
    unfold entails, or; intuition.
  Qed.

  Lemma or_intro1: forall x y z,
    x |- y ->
    x |-  y || z.
  Proof.
    unfold entails, or; intuition.
  Qed.

  Lemma or_intro2: forall x y z,
    x |- z ->
    x |- y || z.
  Proof.
    unfold entails, or; intuition.
  Qed.

  Lemma impl_intro : forall x y z,
    x && y |- z ->
    x |- y => z.
  Proof.
    unfold entails, and, impl; intuition.
  Qed.

  Lemma impl_elim : forall x y z w,
    x |- y => z ->
    w |- y ->
    x && w |- z.
  Proof.
    unfold entails, impl, and; intuition.
  Qed.

  Lemma star_assoc : forall x y z,
    (equ (x * (y * z)) ((x * y) * z))%predicate.
  Proof.
    intros; split; unfold entails, star; firstorder.
    apply join_com in H.
    apply join_com in H1.
    destruct (join_assoc _ _ _ _ _ _ H1 H).
    destruct a0.
    exists x4; exists x3; intuition.
    apply join_com; auto.
    exists x0; exists x2; intuition.
    apply join_com; auto.

    destruct (join_assoc _ _ _ _ _ _ H0 H).
    destruct a0.
    exists x2; exists x4; intuition.
    exists x3; exists x1; intuition.
  Qed.

  Lemma star_com : forall x y, (equ (x * y) (y * x))%predicate.
  Proof.
    intros; split; unfold entails, star; firstorder.
    exists x1; exists x0; intuition; apply join_com; auto.
    exists x1; exists x0; intuition; apply join_com; auto.
  Qed.

  Lemma star_unit1 : forall x, (equ x (emp * x))%predicate.
  Proof.
    intros; split; unfold entails, emp, star; intuition.
    destruct (join_ex_units sa a) as [u Hu].
    exists u; exists a; intuition.
    change (unit_for sa u u).
    rewrite <- (identity_unit_equiv sa u).
    apply unit_identity with a; auto.
    firstorder.
    replace a with x1; auto.
    assert (identity sa x0).
    apply unit_identity with x0; auto.
    apply H2; auto.
  Qed.

  Lemma star_unit2 : forall x, (equ x (x * emp))%predicate.
  Proof.
    intros.
    destruct (star_unit1 x).
    destruct (star_com x emp).
    split; unfold entails in *; intuition.
  Qed.

  Lemma star_intro : forall x y z w,
    x |- z ->
    y |- w ->
    x * y |- z * w.
  Proof.
    unfold entails, star; firstorder.
  Qed.

  Lemma wand_intro : forall x y z,
    x * y |- z ->
    x |- y =* z.
  Proof.
    unfold entails, star, wand; intuition.
    generalize (H a'); firstorder.
    apply H3 with a.
    exists a1; intuition.
    apply join_com; auto.
  Qed.

  Lemma wand_elim : forall x y z w,
    x |- y =* z ->
    w |- y ->
    x * w |- z.
  Proof.
    unfold entails, star, wand; firstorder.
    eapply H; eauto.
    apply join_com; auto.
  Qed.

    Lemma wand_elim2: forall A B, A*(A =* B) |- B.
      intros.
      intro; intros.
      destruct H as [w1 [w2 [? [? ?]]]].
      specialize (H1 w1 _ H H0).
      auto.
    Qed.

    Lemma wand_elim1: forall A B C, A |- B * (B =* C) -> A |- C.
    Proof.
      intros.
      apply entails_trans with (B * (B =* C))%predicate; auto.
      apply wand_elim2.
    Qed.


  Lemma star_dist1: forall A B C D,
    A |- (B && C) * D ->
    A |- (B * D) && (C * D).
  Proof.
    intros. intros w ?.
    destruct H with w as [w1 [w2 [? [[? ?] ?]]]]; auto.
    split; 
    exists w1; exists w2; auto.
  Qed.

    Definition precise A := forall B C, (equ (A * (B && C)) ((A * B) && (A * C)))%predicate.

    Definition pred_world w : formula:= fun w' => w'=w.


  End Facts.

  Module SetoidFacts.
    Export Morphisms.
    Export SetoidTactics.
    Require Export SetoidClass.

    Program Instance SLEquSetoid : Setoid formula :=
      { equiv:= equ }.
    Next Obligation.
      firstorder.
    Qed.

    Add Morphism entails : entails_m.
      unfold entails; firstorder.
    Qed.

    Add Morphism star with signature equiv ==> equiv ==>  equiv  as star_m.
      unfold equ; intros.
      destruct H; destruct H0.
      split; apply star_intro; auto.
    Qed.

    Add Morphism and with signature equiv ==> equiv ==>  equiv  as and_m.
      unfold equ, and; intros.
      destruct H; destruct H0.
      split; intro w; intros; intuition.
    Qed.

    Add Morphism or with signature equiv ==> equiv ==>  equiv as or_m.
      unfold equ, or; intros.
      destruct H; destruct H0.
      split; intro w; intros; intuition.
    Qed.

    Add Morphism impl with signature equiv ==> equiv ==>  equiv as impl_m.
      unfold equ, impl; intros.
      destruct H; destruct H0.
      split; intro w; intros; intuition.
    Qed.

    Add Morphism wand with signature equiv ==> equiv ==> equiv  as wand_m.
      intros.
      split; apply wand_intro.
      rewrite<- H0.
      rewrite<- H.
      rewrite star_com.
      apply wand_elim2.
      rewrite H0.
      rewrite H.
      rewrite star_com.
      apply wand_elim2.
    Qed.

    Add Morphism Notation.forces with signature eq ==> equiv ==> equiv as forces_m.
      intros.
      destruct H.
      split; intros.
      apply H. auto.
      apply H0. auto.
    Qed.
  End SetoidFacts.

  Section MoreFacts.
    Import Notation.
    Import SetoidFacts.
    Open Scope forcing_scope.
    Open Scope entails_scope.
    
    Lemma precise_join_split: forall wa wb wab B C, precise B -> wa |= B -> join sa wa wb wab -> wab |= B*C -> wb |= C.
      intros.
      remember (pred_world wb) as C'.

      assert (wab |=  B*C && (B*C')).
      split; auto.
      rewrite HeqC'.
      exists wa; exists wb; repeat split; auto.
      rewrite<- (H C C') in H3.
      destruct H3 as [wa' [wb' [? [? ?]]]].
      destruct H5.
      rewrite HeqC' in H6.
      substitute H6.
      auto.
    Qed.

  Lemma star_com_eq : forall x y, ((x * y) == (y * x))%predicate.
  Proof.
    intros; split; unfold entails, star; firstorder.
    exists x1; exists x0; intuition; apply join_com; auto.
    exists x1; exists x0; intuition; apply join_com; auto.
  Qed.

  Lemma star_unit1_eq : forall x, (x == (emp * x))%predicate.
  Proof.
    intros; split; unfold entails, emp, star; intuition.
    destruct (join_ex_units sa a) as [u Hu].
    exists u; exists a; intuition.
    change (unit_for sa u u).
    rewrite <- (identity_unit_equiv sa u).
    apply unit_identity with a; auto.
    firstorder.
    replace a with x1; auto.
    assert (identity sa x0).
    apply unit_identity with x0; auto.
    apply H2; auto.
  Qed.

  Lemma star_unit2_eq : forall x, (x == (x * emp))%predicate.
  Proof.
    intros.
    destruct (star_unit1 x).
    destruct (star_com x emp).
    split; unfold entails in *; intuition.
  Qed.


  End MoreFacts.   

  Module Hints.
    Import Notation.
    Import SetoidFacts.
    Open Scope forcing_scope.
    Open Scope entails_scope.
    Hint Unfold pred_world : seplogic_hints.
    Hint Resolve refl top_intro bot_elim  : seplogic_hints.

    Lemma impl_elim': forall x y z : formula, x |- y => z -> x && y |- z.
      intros; eapply impl_elim; eauto with seplogic_hints.
    Qed.

    Lemma wand_elim' : forall x y z : formula, x |- y =* z -> x * y |- z.
      intros. eapply wand_elim; eauto with seplogic_hints.
    Qed.

    Lemma star_com': forall x y x' y', x'|-x -> y'|-y -> x' * y' |- y * x.
      intros. rewrite star_com. apply star_intro; auto.
    Qed.

  Lemma and_elim1' : forall y z, y && z |- y.
  Proof.
    unfold entails, and; intuition.
  Qed.

  Lemma and_elim2' : forall y z, y && z |- z.
  Proof.
    unfold entails, and; intuition.
  Qed.

  Lemma or_intro1': forall y z, y |-  y || z.
  Proof.
    unfold entails, or; intuition.
  Qed.

  Lemma or_intro2': forall y z, z |- y || z.
  Proof.
    unfold entails, or; intuition.
  Qed.

    Hint Resolve and_intro and_elim1' and_elim2' and_elim3 and_elim4: seplogic_hints.
    Hint Resolve or_elim or_intro1' or_intro2'.
    Hint Resolve impl_intro impl_elim' : seplogic_hints.
    Hint Resolve star_assoc star_intro : seplogic_hints.
    Hint Resolve wand_intro wand_elim' wand_elim1 : seplogic_hints.

    Hint Immediate entails_trans : seplogic_ehints.

    Lemma star_unit_l: forall A A', A|-A' -> A |- A'*emp.
      intros. eapply entails_trans; eauto. eapply star_unit2.
    Qed.
    Lemma star_unit_r: forall A A', A|-A' -> A |- emp*A'.
      intros. eapply entails_trans; eauto. eapply star_unit1.
    Qed.

    Hint Transparent equ  : seplogic_hints.
(*    Hint Resolve star_com' : seplogic_hints.*)
    Hint Resolve star_unit_l star_unit_r : seplogic_hints.

  End Hints.

End Soundness.
