(*
 * Copyright 2010, Christian J. Bell
 *
 *)

Require Import base.
Require Import sepalg.
Require Import sepalg_generators.
Require Import sepalg_model.
Require Import HistSet.
Require Import shares.

Set Implicit Arguments.

Lemma top_nonidentity: (~identity Share.sa Share.top).
Proof.
  intro.
  apply Share.shares_nontrivial.
  generalize (Share.bot_identity); intro.
  apply identities_unique with Share.sa; auto.
  unfold identity in *.
  destruct (Share.top_correct Share.bot).
  specialize (H0 _ _ H1).
  destruct (Share.top_correct Share.top).
  specialize (H _ _ H2).
  subst.
  unfold joins.
  exists Share.top.
  apply join_com.
  auto.
Qed.

Section ChannelEndpoint.
  Variables A B:Type.
  Variable ja:sepalg A.
  Variable jb:wsepalg B.

  Definition ch_endpoint_prop (x:A) (y:B) := identity ja x -> identity jb y.

  Inductive ch_endpoint: Type :=
  | ch_endpoint_ab: forall x:A*B, ch_endpoint_prop (fst x) (snd x) -> ch_endpoint.

  Definition ce_pair (x:ch_endpoint) : A*B := match x with ch_endpoint_ab a _ => a end.
  Definition ce_share (x:ch_endpoint) : A := fst (ce_pair x).
  Definition ce_hist (x:ch_endpoint) : B := snd (ce_pair x).

  Definition ch_make_non_unit (a : A) (b : B) (ni : ~ identity ja a):= 
    ch_endpoint_ab (a, b) (fun H : identity ja a => False_ind (identity jb b) (ni H))
     : ch_endpoint.

  Lemma ch_endpoint_eq: forall (x y:A*B) (i1: ch_endpoint_prop (fst x) (snd x)) (i2: ch_endpoint_prop (fst y) (snd y)),
    x=y -> ch_endpoint_ab x i1 = ch_endpoint_ab y i2.
  Proof.
    intros. subst.
    assert (i1 = i2). apply proof_irr.
    subst. trivial.
  Qed.

  Lemma ce_endpoint_share_hist_eq: forall a1 a2, ce_share a1 = ce_share a2 -> ce_hist a1 = ce_hist a2 -> a1=a2.
  Proof.
    intros.
    destruct a1 as [a1 Hp1].
    destruct a2 as [a2 Hp2].
    destruct a1 as [x1 y1].
    destruct a2 as [x2 y2].
    unfold ce_share, ce_hist in *.
    simpl in *.
    subst.
    replace Hp1 with Hp2.
    trivial.
    apply proof_irr.
  Qed.

  Lemma chend_wprod_obligation1: forall x : ch_endpoint, identity wsaf (ce_share x) -> identity jb (ce_hist x).
  Proof. destruct x; auto. Qed.
  Lemma chend_wprod_obligation2: forall (a : A) (b : B),
        (identity wsaf a -> identity jb b) ->
        {x : ch_endpoint &  ce_share x = a /\ ce_hist x = b}.
  Proof. intros; exists (ch_endpoint_ab (a,b) H); auto. Qed.
  Lemma chend_wprod_obligation3: forall x1 x2 : ch_endpoint, ce_share x1 = ce_share x2 -> ce_hist x1 = ce_hist x2 -> x1 = x2.
  Proof. apply ce_endpoint_share_hist_eq; auto. Qed.

  Definition sa_ch_endpoint: sepalg ch_endpoint :=
    sa_wprod _ _ _ _ _ _ _ chend_wprod_obligation1 chend_wprod_obligation2 chend_wprod_obligation3.

  Lemma join_join_wprod: forall e1 e2 e3,
    @sepalg.join _ sa_ch_endpoint e1 e2 e3 <-> (join_wprod _ _ _ _ _ ce_share ce_hist) e1 e2 e3.
  Proof.
    unfold sa_ch_endpoint, sa_wprod.
    destruct saf_wprod.
    simpl. split; auto.
  Qed.

End ChannelEndpoint.


Module Type World_Model_SPEC.
  Parameter cname vname : Type.
  Parameter value: Type.
End World_Model_SPEC.




Module Type World_SepAlg_SPEC (HSV: HistSet_Value_SPEC) (HS: HistSet_SPEC HSV) (WS: World_Model_SPEC).
  Import WS.

  Module HS_WSA := HistSet_WeakSepAlg HSV HS.
  Definition env:= (vname->option (pjoinable (sa_prod value Share.t (sa_equiv value) Share.sa))).
  Definition channels := cname -> option (pjoinable (sa_ch_endpoint Share.sa HS_WSA.wsa_histset)).
  Record world : Type := MakeWorld {
    st: env;
    pch: channels; (* producers *)
    cch: channels  (* consumers *)
  }.

(*  Definition endpoint_unit :=
    ch_endpoint_ab (Share.bot, HS.histset_unit) (fun _ : identity Share.sa Share.bot => HS_WSA.histset_unit_identity).*)
  Parameter empty_world : world.

  Parameter sa_world : sepalg world.

  Axiom empty_world_identity: identity sa_world empty_world.
  Axiom empty_world_joins: forall w, @join _ sa_world w empty_world w.


End World_SepAlg_SPEC.


Module World_SepAlg (HSV: HistSet_Value_SPEC) (HS: HistSet_SPEC HSV) (WS: World_Model_SPEC) <: World_SepAlg_SPEC HSV HS WS.
  Import WS.

  Module HS_WSA := HistSet_WeakSepAlg HSV HS.
  Definition env:= (vname->option (pjoinable (sa_prod value Share.t (sa_equiv value) Share.sa))).
  Definition channels := cname -> option (pjoinable (sa_ch_endpoint Share.sa HS_WSA.wsa_histset)).
  Record world : Type := MakeWorld {
    st: env;
    pch: channels; (* producers *)
    cch: channels  (* consumers *)
  }.

  Definition endpoint_unit :=
    ch_endpoint_ab (Share.bot, HS.histset_unit) (fun _ : identity Share.sa Share.bot => HS_WSA.histset_unit_identity).
  Definition empty_world := MakeWorld (fun x=>None) (fun d=>None) (fun d=>None).

  Section SepAlgEnv.
    Definition sa_env : sepalg env := sa_fun _ _ (sa_lift _).
  End SepAlgEnv.

  Section SepAlgChannels.
    Definition sa_channels : sepalg channels := sa_fun _ _ (sa_lift _).
  End SepAlgChannels.

  Let env_join := @sepalg.join _ sa_env.
  Let ch_join := @sepalg.join _ sa_channels.

  Definition join_world (a b c : world) : Prop :=
    env_join (st a) (st b) (st c) /\ ch_join (pch a) (pch b) (pch c) /\ ch_join (cch a) (cch b) (cch c).

  Lemma saf_world: sepalgfacts join_world.
  Proof.
  apply SepAlgFacts; unfold join_world.
    (* join_eq *)
    intuition. destruct z; destruct z'.
    replace st1 with st0.
    replace pch1 with pch0.
    replace cch1 with cch0.
    trivial.
    eapply join_eq; eauto.
    eapply join_eq; eauto.
    eapply join_eq; eauto.

    (* join_asssoc *)
    intuition. 
    destruct (join_assoc _ _ _ _ _ _ H1 H).
    destruct (join_assoc _ _ _ _ _ _ H0 H2).
    destruct (join_assoc _ _ _ _ _ _ H4 H5).
    destruct a0; destruct a1; destruct a2.
    exists (MakeWorld x x0 x1).
    repeat split; eauto.

    (* join com *)
    intuition.
    apply join_com; auto.
    apply join_com; auto.
    apply join_com; auto.

    (* join_canc *)
    intuition.
    destruct a1; destruct a2.
    replace st1 with st0.
    replace pch1 with pch0.
    replace cch1 with cch0.
    trivial.
    eapply join_canc; eauto.
    eapply join_canc; eauto.
    eapply join_canc; eauto.

    (* join_ex_identities *)
    intuition.
    destruct (join_ex_identities sa_env (st a)) as [x ?].
    destruct (join_ex_identities sa_channels (pch a)) as [y ?].
    destruct (join_ex_identities sa_channels (cch a)) as [z ?].
    exists (MakeWorld x y z).
    repeat split.
    apply identity_unit; simpl; eapply a0.
    apply identity_unit; simpl; eapply a1.
    apply identity_unit; simpl; eapply a2.

    (* join_self *)
    intuition. destruct a; destruct b.
    replace st1 with st0.
    replace pch1 with pch0.
    replace cch1 with cch0.
    trivial.
    apply join_self with sa_channels; auto.
    apply join_self with sa_channels; auto.
    apply join_self with sa_env; auto.
  Qed.

  Definition sa_world : sepalg world := SepAlgFromFacts saf_world.

  Lemma empty_world_identity: identity sa_world empty_world.
  Admitted.

  Lemma empty_world_joins: forall w, @join _ sa_world w empty_world w.
  Admitted. 

End World_SepAlg.


Module World_Model (HSV: HistSet_Value_SPEC) (HS: HistSet_SPEC HSV) (WS: World_Model_SPEC) (WSA : World_SepAlg_SPEC HSV HS WS).
  Import WS.
  Import WSA.

  Module WM_SA <: SEPALG.
    Definition A := world.
    Definition sa := sa_world.
  End WM_SA.

  Module WM_Soundness := Soundness(WM_SA).
  Import WM_Soundness.

  Import HS.
  
  
  Definition produce_endpoint (d:cname) (s:Share.t) (H:HistSet) : formula :=
    fun w => (forall d', d'<>d-> pch w d' = None) /\
      exists pc, pch w d = Some pc /\ ce_share (projT1 pc) = s /\ ce_hist (projT1 pc) = H /\ ce_share (projT1 pc) = Share.bot
      /\ forall d, cch w d = None
      /\ forall x, st w x = None.

  Definition consume_endpoint (d:cname) (s:Share.t) (H:HistSet) : formula :=
    fun w => (forall d', d'<>d-> cch w d' = None) /\
      exists pc, cch w d = Some pc /\ ce_share (projT1 pc) = s /\ ce_hist (projT1 pc) = H /\ ce_share (projT1 pc) = Share.bot
      /\ forall d, pch w d = None
      /\ forall x, st w x = None.

  Definition vown (x:vname) : formula :=
    fun w => (forall x', x'<>x-> st w x' = None)
      /\ exists xvs, st w x = Some xvs
      /\ snd (projT1 xvs) = Share.top
      /\ forall d, pch w d = None
      /\ forall d, cch w d = None.

  Definition vsee (x:vname) : formula :=
    fun w => (forall x', x'<>x-> st w x' = None)
      /\ exists xvs, st w x = Some xvs
      /\ forall d, pch w d = None
      /\ forall d, cch w d = None.

  Definition veq (x:vname) (v:value): formula :=
    fun w => (forall x', x'<>x-> st w x' = None)
      /\ exists xvs, st w x = Some xvs
      /\ fst (projT1 xvs) = v
      /\ forall d, pch w d = None
      /\ forall d, cch w d = None.

  Definition in_histset (h:Hist) (H:HistSet) : formula :=
    fun w => In h H.

  Theorem star_prod: forall d s1 s2 H1 H2, @sepalg.join Share.t Share.sa s1 s2 Share.top ->
    ~identity Share.sa s1 -> ~identity Share.sa s2 ->
    star (produce_endpoint d s1 H1) (produce_endpoint d s2 H2) = produce_endpoint d Share.top (Merge H1 H2).
  Proof.
  Admitted.

  Theorem star_cons: forall d s1 s2 H1 H2, @sepalg.join Share.t Share.sa s1 s2 Share.top ->
    ~identity Share.sa s1 -> ~identity Share.sa s2 ->
    star (consume_endpoint d s1 H1) (consume_endpoint d s2 H2) = consume_endpoint d Share.top (Merge H1 H2).
  Proof.
  Admitted.

  Lemma star_top_elim: forall x, entails x (star x top).
  Proof.
    intros.
    intros w ?.
    exists w.
    exists empty_world.
    repeat split.
    apply empty_world_joins.
    auto.
  Qed.

End World_Model.
