(*
 * Copyright (c) 2009, Andrew Appel, Robert Dockins and Aquinas Hobor.
 *
 *)
(*
 * Modified 2010, Christian J. Bell; extended with history resources
 *
 *)

(** * The definition of multi-unit separation algebras.

    Multi-unit separation algrbras are three place relations
    over some type [A] such that the relation describes
    a partial binary function which is associative, commutitavie
    and cancellative.  We also require the existance of a unit
    for each element, and the "disjointness" property,
    given by [saf_self_join].    
 *)

Require Import base.

Set Implicit Arguments.

Class wsepalg (t : Type) : Type := WeakSepAlg
  { join : t -> t -> t -> Prop;
    wsa_functional: forall x y z z', join x y z -> join x y z' -> z = z';
    wsa_associative: forall a b c d e, join a b d -> join d c e ->
                    {f : t & join b c f /\ join a f e};
    wsa_commutative: forall a b c, join a b c -> join b a c;
    wsa_cancellative: forall a1 a2 b c, join a1 b c -> join a2 b c -> a1=a2;
    wsa_unit_for : t -> t -> Prop := fun e a => join e a a;
    wsa_exist_units: forall a, {e : t & wsa_unit_for e a}
  }.

(** This record contains the axioms of multi-unit separation algebras.
   We use sigma types (basically, a constructive exists) here so that
   we can later avoid using the axiom of choice.
*)
Class sepalgfacts (t : Type) (join : t -> t -> t -> Prop) : Type := SepAlgFacts {
   saf_eq: forall x y z z', join x y z -> join x y z' -> z = z';
   saf_assoc: forall a b c d e, join a b d -> join d c e ->
                    {f : t & join b c f /\ join a f e};
   saf_com: forall a b c, join a b c -> join b a c;
   saf_cancellation: forall a1 a2 b c, join a1 b c -> join a2 b c -> a1=a2;
   saf_unit_for : t -> t -> Prop := fun e a => join e a a;
   saf_exist_units: forall a, {e : t & saf_unit_for e a};
   saf_self_join: forall a b, join a a b -> a = b
}.

(** A separation algebra is a type, a join relation on that type, and
    the above facts about that relation *)
Class sepalg (t:Type) : Type := SepAlg
  { wsaf :> wsepalg t;
    sa_self_join: forall a b, join a a b -> a = b }.

Definition coerce_sepalg_wsepalg (t : Type) (sa : sepalg t) := 
  match sa with SepAlg wsaf0 _ => wsaf0 end.

Definition SepAlgFromFacts (t:Type) (join':t->t->t->Prop) (facts:sepalgfacts join') :=
  match facts with SepAlgFacts A B C D E F => 
    SepAlg (WeakSepAlg join' A B C D E) F
  end.

Coercion coerce_sepalg_wsepalg : sepalg >-> wsepalg.


(** Now we provide "syntactic sugar" to unpack the sepalg type *)
Section WeakSepAlgAxioms.

  (* This will allow use to omit the parameters t and j without explicitly
     using them in every line. *)
  Variable t : Type.
  Variable j : wsepalg t.

  Lemma join_eq : forall x y z z', join x y z -> join x y z' -> z = z'.
  Proof.
    unfold join; intros; destruct j; eauto.
  Qed.

  Lemma join_assoc: forall a b c d e, join a b d -> join d c e ->
                    {f : t & join b c f /\ join a f e}.
  Proof.
    unfold join; intros; destruct j; eauto.
  Qed.

  Lemma join_com: forall a b c, join a b c -> join b a c.
  Proof.
    unfold join; intros; destruct j; eauto.
  Qed.

  Lemma join_canc: forall a1 a2 b c, join a1 b c -> join a2 b c -> a1=a2.
  Proof.
    unfold join; intros; destruct j; eauto.
  Qed.

  (** Usually other proofs do not use this definition and the axiom
     exist_units directly, instead using the definition join_identity 
     and the lemmas join_ex_identities, unit_identity, etc. as 
     defined below *)
  Definition unit_for (e a : t) : Prop := join e a a.

  Lemma join_ex_units: forall a, {e : t & unit_for e a}.
  Proof.
    unfold unit_for, join; destruct j; auto.
  Qed.
End WeakSepAlgAxioms.


Section SepAlgAxioms.

  (* This will allow use to omit the parameters t and j without explicitly
     using them in every line. *)
  Variable t : Type.
  Variable j: sepalg t.

  (** Usually other proofs do not us this axiom directly, instead using
     the lemmas join_split_identity, join_sub_joins_identity, and
     join_overlap as defined below *)
  Lemma join_self: forall a b, join a a b -> a = b.
  Proof.
    unfold unit_for, join; destruct j; auto.
  Qed.
End SepAlgAxioms.

(** * Lemmas about separation algebras. *)

(** Derived definitions and some easy lemmas. *)
Section JoinDerived.
  Variable t : Type.
  Variable j : wsepalg t.

  (** Elements [a] and [b] join. *)
  Definition joins (a b : t) : Prop :=
    exists c, join a b c.

  Definition overlap (a b: t) := ~(joins a b).

  Lemma join_joins: forall a b c,
    join a b c -> joins a b.
  Proof.
    intros; exists c; auto.
  Qed.

  Lemma join_joins': forall a b c,
    join a b c -> joins b a.
  Proof.
    intros.
    exists c. 
    apply join_com.
    trivial.
  Qed.

  Lemma joins_sym: forall a b,
    joins a b = joins b a.
  Proof.
    intros.
    apply prop_ext.
    split; intro; destruct H;
      exists x; apply join_com; trivial.
  Qed.

  (** Elememt [a] is a subelement of [c] .  This relation
      forms a partial order. *)
  Definition join_sub (a c : t) : Prop :=
    exists b, join a b c.

  Lemma join_join_sub: forall a b c,
    join a b c ->
    join_sub a c.
  Proof.
    intros. exists b; auto.
  Qed.

  Lemma join_join_sub': forall a b c,
    join a b c ->
    join_sub b c.
  Proof.
    intros.
    exists a.
    apply join_com.
    trivial.
  Qed.

  Lemma join_sub_refl: forall a,
    join_sub a a.
  Proof.
    intro.
    destruct (join_ex_units j a).
    exists x.
    apply join_com.
    trivial.
  Qed.

  Lemma join_sub_trans: forall a b c,
    join_sub a b ->
    join_sub b c ->
    join_sub a c.
  Proof.
    intros.
    destruct H; destruct H0.
    generalize (join_assoc _ _ _ _ _ _ H H0); intros.
    destruct X.
    destruct a0.
    exists x1.
    auto.
  Qed.

  (** This lemma does not use join_self, but is used later in section JoinSelf. *)
  Lemma join_sub_joins: forall a b,
    join_sub a b -> joins a b -> joins a a.
  Proof.
    intros.
    destruct H; destruct H0.
    apply join_com in H0.
    destruct (join_assoc j _ _ _ _ _ H H0) as [? [? ?]].
    apply join_com in H2.
    destruct (join_assoc j _ _ _ _ _ H1 H2) as [? [? ?]].
    exists x2; auto.
  Qed.
  (* Note: there are two other conceivable conclusions from the above 
     premises: "joins b b" and "join_sub b a".  Neither must follow, since
     neither is true on the bools, but also neither is a contradiction 
     since both are true on Z *)
  
  Lemma join_sub_joins_trans: forall a b c,
    join_sub a c -> joins c b -> joins a b.
  Proof.
    intros.
    destruct H as [wx ?].
    destruct H0 as [wy ?].
    destruct (join_assoc _ _ _ _ _ _ (join_com _ _ _ _ H) H0) as [wf [? ?]].
    econstructor; eauto.
  Qed.

End JoinDerived.

Hint Resolve join_joins join_joins' join_join_sub join_join_sub'.

(** Now we will move from the definition of unit, which is hard to use,
   to the notion of identity, which is more powerful. *)
Section JoinIdentity.
  Variable t : Type.
  Variable j : wsepalg t.

  (** Unlike the definition of unit, an identity is a unit for everything
     with which it joins; our goal will be to show that the two definitions
     are equal. *)
  Definition identity: t -> Prop := 
    fun e => forall a b, join e a b -> a = b.

  Definition nonidentity (a: t) := ~(identity a).

  (** If [a] is a subelement of [b], their units are equal. *)
  Lemma join_sub_units_eq: forall a b ea eb,
    join_sub j a b ->
    unit_for j ea a ->
    unit_for j eb b ->
    ea = eb.
  Proof.
    unfold unit_for.
    intros.
    destruct H.
    destruct (join_assoc _ _ _ _ _ _ H0 H) as [f [? ?]].
    generalize (join_eq _ _ _ _ _ H H2); intro.
    subst f.
    eapply join_canc; eauto.
  Qed.

  (** A unit for an element is a unit for itself (is idempotent). *)
  Lemma unit_self_unit: forall a ea,
    unit_for j ea a ->
    unit_for j ea ea.
  Proof.
    intros.
    generalize (join_join_sub _ _ _ _ H); intro.
    destruct (join_ex_units j ea) as [ee H1].
    generalize (join_sub_units_eq _ _ H0 H1 H); intro.
    subst ea.
    trivial.
  Qed.

  (** If a joins with b, then their units are equal. *)
  Lemma joins_units_eq: forall a b ea eb,
    joins j a b ->
    unit_for j ea a ->
    unit_for j eb b ->
    ea = eb.
  Proof.
    intros. destruct H.
    destruct (join_ex_units j x).
    assert (join_sub j a x) by eauto. (* coerce? *)
    generalize (join_sub_units_eq _ _ H2 H0 u).
    assert (join_sub j b x) by (apply join_com in H; eauto).
    generalize (join_sub_units_eq _ _ H3 H1 u).
    congruence.
  Qed.

  (** The existance of identity elements. *)
  Lemma join_ex_identities: forall a, 
    {e : t & identity e /\ joins j e a}.
  Proof.
    intro x.
    destruct (join_ex_units j x) as [e H0].
    exists e.
    split; eauto.
    generalize (unit_self_unit _ _ H0); intro.
    clear -H.
    repeat intro.
    destruct (join_ex_units j a) as [ea H1].
    generalize (join_joins _ _ _ _ H0); intro H2.
    generalize (joins_units_eq _ _ H2 H H1); intro.
    subst ea.
    eapply join_eq; eauto.
  Qed.

  (** If something is a unit then it is an identity. *)
  Lemma unit_identity: forall e a,
    unit_for j e a -> identity e.
  Proof.
    intros.
    destruct (join_ex_identities a) as [ea [? [b ?]]].
    generalize (H0 a b H1); intro.
    subst b.
    generalize (join_canc _ _ _ _ _ H H1); intro.
    subst ea.
    trivial.
  Qed.

  (** If something is an identity and it joins with an element then it is a 
     unit for that element. *) 
  Lemma identity_unit: forall e a,
    identity e ->
    joins j e a ->
    unit_for j e a.
  Proof.
    intros. destruct H0.
    generalize (H a x H0); intro.
    subst x.
    trivial.
  Qed.

  (** Identities are exactly units for themselves (are idempotent).*)
  Lemma identity_unit_equiv: forall a,
    (identity a) = (unit_for j a a).
  Proof.
    intros.
    apply prop_ext; split; intro.
    apply identity_unit; trivial.
    exists a.
    destruct (join_ex_units j a) as [ea H0].
    generalize (join_com _ _ _ _ H0); intro.
    generalize (H ea a H1); intro.
    subst ea.
    trivial.
    apply (unit_identity _ _ H).
  Qed.

  (** Joinable identities are unique. *)
  Lemma identities_unique : forall e1 e2,
    identity e1 ->
    identity e2 ->
    joins j e1 e2 ->
    e1 = e2.
  Proof.
    intros. destruct H1.
    assert (e2 = x) by auto.
    apply join_com in H1.
    assert (e1 = x) by auto.
    congruence.
  Qed.

End JoinIdentity.

(** Finally, we give the lemmas that cover the typical use of the self_join axiom.
    All of the rest of the lemmas require the self_join axiom.
  *)
Section JoinSelf.
  Variable t : Type.
  Variable j : sepalg t.

  (** This lemma gives a weaker property than the full self_join axiom.
      Example: This axiom holds on N but eliminates Z, while the full axiom
      also eliminates N.  This statement is equivalant to the positivity
      axiom. *)
  Lemma split_identity: forall a b c,
    join a b c -> identity j c -> identity j a.
  Proof.
    intros.
    rewrite identity_unit_equiv in *.
    destruct (join_assoc j _ _ _ _ _ H H0) as [? [? ?]].
    generalize (join_com j _ _ _ H1); intro.
    destruct (join_assoc j _ _ _ _ _ H H3) as [? [? ?]].
    generalize (join_self j _ _ H4); intro.
    subst x0.
    assert (unit_for j b b) by trivial.
    rewrite <- identity_unit_equiv in H6.
    generalize (join_com j _ _ _ H); intro.
    generalize (H6 _ _ H7); intro.
    subst c.
    trivial.
  Qed.

  (* The contrapositive of split_identity *)
  Lemma join_nonidentity: forall a b c,
    nonidentity j a -> join a b c -> nonidentity j c.
  Proof.
    intros a b c H H0 H1.
    contradiction H.
    apply split_identity with b c; auto.
  Qed.
  
  (** This lemma only requires split_identity. *)
  Lemma join_sub_antisym : forall x y, 
    join_sub j x y -> 
    join_sub j y x -> 
    x = y.
  Proof.
    firstorder.
    destruct (join_assoc j _ _ _ _ _ H H0); intuition.
    apply join_com in H2.
    generalize(unit_identity _ _ _ H2); intro.
    generalize(split_identity _ _ H1 H3); intro.
    apply join_com in H.
    auto.
  Qed.

  (** This lemma uses the full power of self_join and eliminates both N and Z. *)
  Lemma join_sub_joins_identity: forall a b,
    join_sub j a b -> joins j a b -> identity j a.
  Proof.
    intros.
    rewrite identity_unit_equiv.
    generalize (join_sub_joins H H0); intro.
    destruct H1.
    generalize (join_self j _ _ H1); intro.
    subst x.
    trivial.
  Qed.

  (** Sometimes it is useful to use a negative form of the previous lemma *)
  Lemma join_overlap: forall a b,
    join_sub j a b -> nonidentity j a -> overlap j a b.
  Proof.
    repeat intro.
    apply H0.
    eapply join_sub_joins_identity; eauto.
  Qed.

End JoinSelf.

(** The elements of a multi-unit separation algebra can be partitioned
    into equivalance classes, where two elements are in the class iff
    they have the same unit.
  *)
Section JoinEquivalence.
  Variable t : Type.
  Variable j : wsepalg t.

  Definition comparable (a b:t)
    := projT1 (join_ex_identities j a) = projT1 (join_ex_identities j b).

  Lemma comparable_refl : forall a, comparable a a.
  Proof. intros; red; auto. Qed.

  Lemma comparable_sym : forall a b, comparable a b -> comparable b a.
  Proof. unfold comparable; auto. Qed.

  Lemma comparable_trans : forall a b c, comparable a b -> comparable b c -> comparable a c.
  Proof. unfold comparable; intros; eapply trans_eq; eauto. Qed.

  Lemma comparable_common_unit : forall a b, 
    comparable a b ->
    exists e, join e a a /\ join e b b.
  Proof.
    intros.
    red in H.
    exists (projT1 (join_ex_identities j a)); split.
    clear.
    destruct (join_ex_identities j a).
    apply identity_unit; simpl; firstorder.
    rewrite H.
    destruct (join_ex_identities j b).
    clear -a0. (* Get around a bug in Coq *)
    apply identity_unit; simpl; firstorder.
  Qed.

  Lemma common_unit_comparable : forall a b,
    (exists e, join e a a /\ join e b b) ->
    comparable a b.
  Proof.
    intros.
    destruct H as [e [H1 H2]].
    unfold comparable.
    destruct (join_ex_identities j a).
    destruct (join_ex_identities j b).
    simpl.
    replace x with e; firstorder.
    apply join_canc with j b b; auto.
    replace x2 with b in H4; auto.
    replace x1 with a in H0; auto.
    apply join_canc with j a a; auto.
  Qed.

End JoinEquivalence.
