(*
 * Copyright (c) 2009, Andrew Appel, Robert Dockins and Aquinas Hobor.
 *
 *)
(*
 * Modified 2010, Christian J. Bell; extended with history resources
 *
 *)

(** This module defines the standard operators on separation algebras, including
    the operators over pairs, disjoint sums, function spaces, dependent products,
    dependent sums, sub-separation algebras, the discrete separation algebra,
    the lifting operator, and the trivial unit and void separation algrbras.
*)

Require Import base.
Require Import sepalg.

(** The trivial separation algebra over the unit type.  This SA
    is the identity of the product SA operator, up to isomorphism.
*)
Section SepAlgUnit.
  Definition join_unit (x y z:unit) := True.

  Lemma unit_eq : forall x y:unit, x = y.
    intros; case x; case y; trivial.
   Qed.

  Lemma saf_unit : sepalgfacts join_unit.
  Proof.
  apply SepAlgFacts; firstorder; apply unit_eq.
  Qed.

  Definition sa_unit : sepalg unit := SepAlgFromFacts saf_unit.
End SepAlgUnit.

(** The trivial separation algrbra over the void type. This SA
    is the identity of the coproduct (disjoint sum) SA operator, up to isomorphism.
*)
Section SepAlgVoid.
  Inductive Void : Type :=.

  Definition join_void (x y z:Void) := False.

  Lemma saf_void : sepalgfacts join_void.
  Proof.
  apply SepAlgFacts; intuition.
  Qed.

  Definition sa_void : sepalg Void := SepAlgFromFacts saf_void.
End SepAlgVoid.

(** The "equivalance" or discrete SA.  In this SA, every element of an arbitrary
    set is made an idempotent element.
*)

Section SepAlgEquiv.
  Variable A:Type.

  Definition join_equiv (x y z:A) := x=y /\ y=z.

  Lemma saf_equiv : sepalgfacts join_equiv.
  Proof.
  apply SepAlgFacts; unfold join_equiv.
    (* join_eq *)
    intuition; congruence.

    (* join_asssoc *)
    intuition; exists a; intuition congruence.

    (* join com *)
    intuition; congruence.

    (* join_canc *)
    intuition congruence.

    (* join_ex_identities *)
    intro a; exists a; auto.

    (* join_self *)
    intuition.
  Qed.

  Definition sa_equiv : sepalg A := SepAlgFromFacts saf_equiv.
End SepAlgEquiv.

(** The subSA operator.  Given some SA and a property [P] on the elements of that SA,
    form the SA consisting of just those elements satisfying [P].  The property
    [P] must be compatible with the original SA, as specified in the hypothesis
    [HPunit] and [HPjoin].
*)
Section SepAlgProp.
  Variable A:Type.
  Variable j:sepalg A.
  Variable P:A -> Prop.

  Hypothesis HPunit :
    forall x e, P x -> unit_for j e x -> P e.

  Hypothesis HPjoin : forall x y z,
    join x y z -> P x -> P y -> P z.

  Definition join_prop (x y z: { a:A & P a }) :=
    join (projT1 x) (projT1 y) (projT1 z).

  Lemma saf_prop : sepalgfacts join_prop.
  Proof.
    apply SepAlgFacts.

    (* join_eq *)
    intros.
    red in H.
    red in H0.
    destruct x; destruct y; destruct z; destruct z'; simpl in *.
    revert p2.
    replace x2 with x1; intros.
    replace p2 with p1; auto.
    apply proof_irr.
    apply join_eq with j x x0; auto.

    (* join_assoc *)
    intros.
    red in H.
    red in H0.
    destruct a; destruct b; destruct c; destruct d; destruct e.
    simpl in *.
    unfold join_prop; simpl.
    destruct (join_assoc j _ _ _ _ _ H H0) as [f [? ?]].
    assert (P f).
    apply HPjoin with x0 x1; auto.
    exists (existT (fun x => P x) f H3).
    simpl; auto.

    (* join_com *)
    intros; red; apply join_com; auto.

    (* join_canc *)
    intros.
    destruct a1; destruct a2; destruct b; destruct c.
    red in H; red in H0; simpl in *.
    revert p0.
    replace x0 with x; intros.
    replace p0 with p; auto.
    apply proof_irr.
    apply join_canc with j x1 x2; auto.
    
    (* join_ex_identities *)
    intros.
    destruct a as [x Hx].
  
    destruct (join_ex_identities j x) as [e He].
    assert (P e) by (apply HPunit with x; trivial; apply identity_unit; firstorder).
    exists (existT (fun x => P x) e H).
    red; simpl; apply identity_unit; firstorder.
    exists x0; auto.

    (* join_self *)
    intros.
    destruct a; destruct b.
    red in H; simpl in H.
    revert p0.
    replace x0 with x; intros.
    replace p0 with p; auto.
    apply proof_irr.
    apply join_self with j; auto.
  Qed.

  Definition sa_prop : sepalg { a:A & P a } := SepAlgFromFacts saf_prop.
End SepAlgProp.

(** The function space operator from a key type [key] to
    a separtion algebra on type [t'].
*)

Section SepAlgFun.
  Variable key: Type.
  Variable t' : Type.
  Variable range: sepalg t'.

  Definition m_join (a b c : key -> t') : Prop := forall x, join (a x) (b x) (c x).

  Lemma saf_fun: sepalgfacts m_join.
  Proof.
    apply SepAlgFacts.

    (* join_eq *)
    unfold m_join; intros; extensionality a; apply (join_eq range) with (x a) (y a); auto.

    (* join_assoc *)
    unfold m_join; intros.
    set (F := fun x => join_assoc range _ _ _ _ _ (H x) (H0 x)).
    exists (fun x => projT1 (F x)); split; intros; destruct (F x); simpl; intuition.

    (* join_com *)
    unfold m_join; intros; apply (join_com range); auto.

    (* join_canc *)
    unfold m_join; intros; extensionality x; intros; apply (join_canc range) with (b x) (c x); auto.

    (* join_ex_identities *)
    unfold m_join; intros.
    set (F := fun x => join_ex_identities range (a x)).
    exists (fun x => projT1 (F x)).
    intro x; destruct (F x); apply identity_unit; simpl; firstorder.
    exists x1; auto.

    (* join_self *)
    unfold m_join; intros.
    unfold join.
    extensionality x.
    apply (join_self range); auto.
  Qed.

  Definition sa_fun : sepalg (key -> t') := SepAlgFromFacts saf_fun.
End SepAlgFun.

(** The dependent product SA operator from an index set [I] into
    a SA indexed by [Pi_j].  It happens that the construction of this
    operator either requires constructive witnesses for the unit
    and associativity axioms (as defined above) or the axiom of choice.
*)

Section SepAlgPi.
  Variable I:Type.  
  Variable Pi: I -> Type.
  Variable Pi_j : forall i:I, sepalg (Pi i).

  Let P := forall i:I, Pi i.

  Definition join_pi (x y z: P) : Prop :=
    forall i:I, join (x i) (y i) (z i).

  Lemma saf_pi : sepalgfacts join_pi.
    constructor.
    
    (* join_eq *)
    intros; apply dep_extensionality; intro i.
    apply (join_eq (Pi_j i) (x i) (y i)); auto.

    (* join_assoc *)
    intros.
    pose (f := fun i => projT1 (join_assoc (Pi_j i) _ _ _ _ _ (H i) (H0 i))).
    exists f.
    split; intro i; unfold f;
      destruct (join_assoc (Pi_j i) _ _ _ _ _ (H i) (H0 i));
      simpl; intuition.

    (* join_com *)
    intros; intro i; apply join_com; auto.

    (* cancellation *)
    intros; apply dep_extensionality; intro i.
    apply (join_canc (Pi_j i)) with (b i) (c i); auto.

    (* exist_identities *)
    intros.
    pose (e := fun i => projT1 (join_ex_units (Pi_j i) (a i))).
    exists e.
    intro i; unfold e.
    destruct (join_ex_units (Pi_j i) (a i)); simpl; auto.

    (* self_join_identity *)
    intros.
    apply dep_extensionality; intro i.
    apply (join_self (Pi_j i) _ _ (H i)).
  Qed.

  Definition sa_pi : sepalg P := SepAlgFromFacts saf_pi.

End SepAlgPi.
Implicit Arguments sa_pi.

(** The dependent sum operator on SAs.

   Here we have defined the operator
   assuming a decidable equality on the index type [I].  This avoids
   having to assume Streicher's K axiom (or one of its equivalants).
*)

Section SepAlgSigma.
  Variable I:Type.
  Variable Sigma: I -> Type.
  Variable Sigma_j : forall i:I, sepalg (Sigma i).

  Variable eq_dec : forall x y:I, {x=y} + {x<>y}.

  Let S := { i:I & Sigma i }.
  Let injS := existT (fun x => Sigma x).

  Require Import Eqdep_dec.

  Let inj_sig_I :=
    eq_dep_eq__inj_pairT2 I (eq_rect_eq__eq_dep_eq I (eq_rect_eq_dec eq_dec)).

  Ltac inj_sig :=
    repeat
    match goal with 
     [ X : (@eq (sigT (fun x => Sigma x)) (existT _ _ ?a) (existT _ _ ?b)) |- _ ] =>
       generalize (inj_sig_I _ _ _ _ X); intros; clear X
    end.

  Inductive join_sigma : S -> S -> S -> Prop :=
    j_sig_def : forall (i:I) (a b c:Sigma i),
      join a b c ->
      join_sigma (injS i a) (injS i b) (injS i c).

  Lemma saf_sigma : sepalgfacts join_sigma.
  Proof.
    apply SepAlgFacts.

    (* join_eq *)
    intros.
    inv H; inv H0; inj_sig; subst.
    replace c with c0; auto.
    apply join_eq with (Sigma_j i) a b; auto.

    (* join_assoc *)
    intros [ai a] [bi b] [ci c] [di d] [ei e]; intros.
    assert (ai = bi /\ bi = ci /\ ci = di /\ di = ei).
    inv H; inv H0; simpl; auto.
    decompose [and] H1; subst; clear H1.
    rename ei into i.
    assert (join a b d).
    inversion H; inj_sig; subst; auto.
    assert (join d c e).
    inversion H0; inj_sig; subst; auto.
    destruct (join_assoc (Sigma_j i) _ _ _ _ _ H1 H2) as [f [? ?]].
    exists (existT (fun i => Sigma i) i f).
    split; constructor; auto.

    (* join_com *)
    intros.
    inv H; inj_sig; subst.
    constructor.
    apply join_com; auto.

    (* join_canc *)
    intros.
    inv H; inv H0; inj_sig; subst.
    replace a with a1; auto.
    apply join_canc with (Sigma_j i) b0 c0; auto.

    (* join_ex_identities *)
    intros [ai a].
    destruct (join_ex_identities (Sigma_j ai) a) as [e ?].
    exists (existT (fun i => Sigma i) ai e).
    constructor; auto.
    apply identity_unit; firstorder.
    exists x; auto.

    (* join_self *)
    intros; inv H; inj_sig; subst.
    replace c with a0; auto.
    apply join_self with (Sigma_j i); auto.
  Qed.

  Definition sa_sigma : sepalg S := SepAlgFromFacts saf_sigma.
End SepAlgSigma.
Implicit Arguments sa_sigma.

(** The SA operator on cartesian products *)
Section SepAlgProd.
  Variables A B:Type.
  Variables (ja:sepalg A) (jb:sepalg B).

  Definition join_prod (x y z:A*B) : Prop :=
    join (fst x) (fst y) (fst z) /\
    join (snd x) (snd y) (snd z).

  Lemma saf_prod : sepalgfacts join_prod.
  Proof.
  apply SepAlgFacts.

    (* join_eq *)
    intuition; destruct z'; simpl in *.
    replace a2 with a1.
    replace b2 with b1.
    trivial.
    
    apply join_eq with jb b b0; firstorder.
    apply join_eq with ja a a0; firstorder.

    (* join_assoc *)
    intuition; firstorder. destruct e; simpl in *.
    destruct (join_assoc ja _ _ _ _ _ H H0) as [x [? ?]].
    destruct (join_assoc jb _ _ _ _ _ H1 H2) as [y [? ?]].
    exists (x,y); firstorder.

    (* join_com *)
    intuition; firstorder; destruct c; simpl in *.
    apply join_com; auto.
    apply join_com; auto.

    (* join_canc *)
    intuition; firstorder; destruct c; simpl in *.
    replace a0 with a.
    replace b0 with b.
    trivial.
    apply (join_canc jb _ _ _ _ H1 H2).
    apply (join_canc ja _ _ _ _ H H0).

    (* join_ex_identities *)
    intuition; firstorder.
    destruct (join_ex_identities ja a0) as [x ?].
    destruct (join_ex_identities jb b) as [y ?].
    exists (x,y); firstorder; apply identity_unit; firstorder.
    exists x0; auto.
    exists x1; auto.

    (* join_self *)
    intuition; firstorder; destruct b0; simpl in *.
    rewrite (join_self ja a0 a); auto.
    rewrite (join_self jb b b0); auto.
  Qed.

  Definition sa_prod : sepalg (A*B) := SepAlgFromFacts saf_prod.
End SepAlgProd.

Section SepAlgWeakProd.
  Variables A B:Type.
  Variables (ja:sepalg A) (jb:wsepalg B).
  Variable X : Type.
  Variable fstX : X -> A.
  Variable sndX : X -> B.
  Variable XP: forall x, identity _ (fstX x) -> identity _ (sndX x).
  Variable MakeX: forall a b, (identity _ a -> identity _ b) -> {x:X & fstX x=a /\ sndX x=b}.
  Variable Xeq: forall x1 x2, fstX x1=fstX x2->sndX x1=sndX x2->x1=x2.

  Definition join_wprod (x y z:X) : Prop :=
    join (fstX x) (fstX y) (fstX z) /\
    join (sndX x) (sndX y) (sndX z).

  Lemma saf_wprod : sepalgfacts join_wprod.
  Proof.
  apply SepAlgFacts.

    (* join_eq *)
    intros.
    unfold join_wprod in *.
    destruct H; destruct H0.
    apply Xeq.
    apply join_eq with ja (fstX x) (fstX y); auto.
    apply join_eq with jb (sndX x) (sndX y); auto.

    (* join_assoc *)
    intuition; firstorder.
    destruct (join_assoc ja _ _ _ _ _ H H0) as [x [? ?]].
    destruct (join_assoc jb _ _ _ _ _ H1 H2) as [y [? ?]].
    destruct (MakeX x y); intros.
    intro; intros.
    apply (XP b).
    apply split_identity with (fstX c) x; auto.
    assert (identity _ (sndX c)).
      apply (XP c).
      apply split_identity with (fstX b) x; auto.
      apply join_com; auto.
    apply join_com in H5.
    rewrite (H9 _ _ H5).
    auto.

    destruct a0; subst.
    exists (x0).
    unfold join_wprod.
    split; auto.

    (* join_com *)
    intuition; destruct H; split; apply join_com; auto.

    (* join_canc *)
    intuition; firstorder.
    apply Xeq.
    apply (join_canc ja _ _ _ _ H H0).
    apply (join_canc jb _ _ _ _ H1 H2).

    (* join_ex_identities *)
    intuition; firstorder.
    destruct (join_ex_identities ja (fstX a)) as [x Ha].
    destruct (join_ex_identities jb (sndX a)) as [y Hb].
    destruct Ha; destruct Hb.
    destruct (MakeX x y) as [x0 [H3 H4]]; intros; auto.
    exists x0. subst.
    unfold join_wprod.
    split.
    destruct H0; replace x with (fstX a) in H0; auto.
    destruct H2; replace x with (sndX a) in H2; auto.

    (* join_self *)
    intuition; firstorder.
    generalize (join_self _ _ _ H); intro.
    substitute<- H1.
    apply Xeq; auto.
    assert (identity jb (sndX a)).
      apply XP.
      apply unit_identity with (fstX a); auto.
    auto.
  Qed.

  Definition sa_wprod : sepalg X := SepAlgFromFacts saf_wprod.
End SepAlgWeakProd.


(** The SA operator on disjoint sums. *)
Section SepAlgSum.
  Variables A B:Type.
  Variables (ja:sepalg A) (jb:sepalg B).

  Definition join_sum (x y z:A+B) : Prop :=
    match x, y, z with
    | inl xa, inl ya, inl za => join xa ya za
    | inr xb, inr yb, inr zb => join xb yb zb
    | _, _, _ => False
    end.

  Lemma saf_sum : sepalgfacts join_sum.
  Proof.
  apply SepAlgFacts.

    (* join_eq *)
    intuition; destruct z'; simpl in *; try contradiction.
    replace a2 with a1; [ trivial | apply (join_eq ja _ _ _ _ H H0) ].
    replace b2 with b1; [ trivial | apply (join_eq jb _ _ _ _ H H0) ].

    (* join_assoc *)
    intuition; destruct e; simpl in *; try contradiction.
    destruct (join_assoc ja _ _ _ _ _ H H0) as [a' [? ?]].
    exists (inl B a'); simpl; intuition.
    destruct (join_assoc jb _ _ _ _ _ H H0) as [b' [? ?]].
    exists (inr A b'); simpl; intuition.

    (* join_com *)
    intuition; destruct c; simpl in *; try contradiction.
    apply join_com; auto.
    apply join_com; auto.

    (* join_canc *)
    intuition; destruct c; simpl in *; try contradiction.  
    replace a0 with a; trivial.
    apply (join_canc ja _ _ _ _ H H0).
    replace b0 with b; trivial.
    apply (join_canc jb _ _ _ _ H H0).

    (* join_ex_identities *)
    intuition.
    destruct (join_ex_identities ja a0).
    exists (inl B x); simpl; apply identity_unit; destruct a; auto. (* firstorder -- bug in Coq *)
    destruct (join_ex_identities jb b).
    exists (inr A x); simpl; apply identity_unit; destruct a; auto. (* firstorder -- bug in Coq *)

    (* join_self *)
    intuition; try destruct b; try destruct b0;
       simpl in *; try contradiction.
    rewrite (join_self ja a0 a); auto.
    rewrite (join_self jb b b0); auto.
  Qed.

  Definition sa_sum : sepalg (A + B) := SepAlgFromFacts saf_sum.
End SepAlgSum.

(** This operator transforms a SA into an isomorphic one by
    passing through the bijection [bij].
*)
Section SepAlgBijection.
  Variables A B:Type.

  Record bijection : Type := Bijection {
     bij_f: A -> B;
     bij_g: B -> A;
     bij_fg: forall x, bij_f (bij_g x) = x;
     bij_gf: forall x, bij_g (bij_f x) = x
  }.

  Variable bij : bijection.

  Variable saa : sepalg A.

  Definition join_bijection (x y z : B) : Prop :=
  match bij with Bijection _ g _ _ => join (g x) (g y) (g z) end.

  Lemma saf_bijection : sepalgfacts join_bijection.
  Proof.
  unfold join_bijection in *.
  destruct bij as [f g fg gf].
  apply SepAlgFacts; intros; auto.

  generalize (f_equal f (join_eq saa _ _ _ _ H H0)); intro; repeat rewrite fg in H1; auto.
  destruct (join_assoc saa _ _ _ _ _ H H0); exists (f x); rewrite gf; auto.
  apply join_com; auto.
  generalize (f_equal f (join_canc saa _ _ _ _ H H0)); repeat rewrite fg; auto.
  destruct (join_ex_identities saa (g a)); exists (f x); rewrite gf; apply identity_unit; firstorder. exists x0; auto.
  generalize (f_equal f (join_self saa _ _ H)); repeat rewrite fg; auto.
  Qed.

  Definition sa_bijection : sepalg B := SepAlgFromFacts saf_bijection.
End SepAlgBijection.

(** This section defines the "strictly positive" ([pjoinable]) elements of
    a SA as the nonidentity elements.  These elements satisfy all the
    SA axioms excepth the existance of identities.

    Next we define the lifting operator by adding a new, unique identity
    element to the strictly positive elements.
*)
Section PJoin.
  Variable T : Type.
  Variable j : sepalg T.

  Definition pjoinable : Type := {t : T & ~ identity j t}.

  Definition pjoin (j1 j2 j3 : pjoinable) : Prop :=
  match (j1, j2, j3) with 
  (existT t1 _, existT t2 _, existT t3 _) => join t1 t2 t3
  end.

  Lemma pjoin_eq: forall x y z z', pjoin x y z -> pjoin x y z' -> z = z'.
  Proof.
  intros.
  destruct x.
  destruct y.
  destruct z.
  destruct z'.
  unfold pjoin in *.
  generalize (join_eq j _ _ _ _ H H0); intro.
  subst x2.
  assert (n1 = n2) by apply proof_irr.
  subst n2.
  trivial.
  Qed.

  Lemma pjoin_assoc: forall a b c d e, pjoin a b d -> pjoin d c e ->
                    {f : pjoinable & pjoin b c f /\ pjoin a f e}.
  Proof.
  intros.
  destruct a.
  destruct b.
  destruct c.
  destruct d.
  destruct e.
  unfold pjoin in *.
  destruct (join_assoc j _ _ _ _ _ H H0) as [? [? ?]].
  assert (~ identity j x4).
  intro.
  clear -H3 H1 n0 n1.
  contradiction n0.
  eapply split_identity; eauto.
  exists (existT (fun t : T => ~ identity j t) x4 H3).
  firstorder.
  Qed.

  Lemma pjoin_com: forall a b c, pjoin a b c -> pjoin b a c.
  Proof.
  intros.
  destruct a.
  destruct b.
  destruct c.
  unfold pjoin in *.
  apply join_com.
  trivial.
  Qed.

  Lemma pcancellation: forall a1 a2 b c, pjoin a1 b c -> pjoin a2 b c -> a1=a2.
  Proof.
  intros.
  destruct a1.
  destruct a2.
  destruct b.
  destruct c.
  unfold pjoin in *.
  generalize (join_canc _ _ _ _ _ H H0); intro.
  subst x0.
  assert (n = n0) by apply proof_irr.
  subst n0.
  trivial.
  Qed.

  Lemma pnidentity: forall p p', pjoin p p' p' -> False.
  Proof.
  intros.
  destruct p.
  destruct p'.
  unfold pjoin in H.
  generalize (unit_identity _ _ _ H).
  contradiction.
  Qed.

  Lemma pnjoin_self: forall p p', pjoin p p p' -> False.
  Proof.
  intros.
  destruct p.
  destruct p'.
  unfold pjoin in H.
  generalize (join_self _ _ _ H); intro.
  subst x0.
  rewrite identity_unit_equiv in *.
  contradiction.
  Qed.

  Inductive lift_join : option pjoinable -> option pjoinable -> option pjoinable -> Prop :=
  | lj_None1 : forall x, lift_join None x x
  | lj_None2 : forall x, lift_join x None x
  | lj_Sime  : forall x y z:pjoinable,
                  pjoin x y z ->
                  lift_join (Some x) (Some y) (Some z).

  Lemma saf_lift : sepalgfacts lift_join.
  Proof.
    constructor.
  
    (* join_eq *)
    intros; inv H; inv H0; auto.
    replace z0 with z; auto.
    eapply pjoin_eq; eauto.

    (* join_assoc *)
    intros.
    destruct b.
    2: exists c; split; try constructor; auto.
    2: inv H; auto.
    destruct c.
    2: exists (Some p); split; try constructor.
    2: inv H0; auto.
    destruct a.
    2: exists e; split; try constructor.
    2: inv H; auto.
    destruct d.
    2: elimtype False; inv H.
    destruct e.
    2: elimtype False; inv H0.
    destruct (pjoin_assoc p1 p p0 p2 p3).
    inv H; auto.
    inv H0; auto.
    exists (Some x); intuition; constructor; auto.

    (* join_com *)
    intros; inv H; constructor.
    apply pjoin_com; auto.

    (* join_canc *)
    intros; inv H; inv H0; auto.
    elim (pnidentity _ _ H3).
    elim (pnidentity _ _ H1).
    replace x0 with x; auto.
    eapply pcancellation; eauto.

    (* join_ex_identity *)
    intros.
    exists None; constructor.

    (* self_join *)
    intros; inv H; auto.
    replace z with x; auto.
    elim (pnjoin_self _ _ H2).
  Qed.

  Definition sa_lift : sepalg (option pjoinable)
    := SepAlgFromFacts saf_lift.

End PJoin.

Implicit Arguments pjoinable.
Implicit Arguments pjoin.
Implicit Arguments pjoin_eq.
Implicit Arguments pjoin_assoc.
Implicit Arguments pjoin_com.
Implicit Arguments pcancellation.

Implicit Arguments sa_lift.
