
#include "stm32f3xx_hal.h"
#include "usb_device.h"
#include "usbd_cdc_if.h"

extern ADC_HandleTypeDef hadc1;
extern ADC_HandleTypeDef hadc2;
extern ADC_HandleTypeDef hadc3;
extern ADC_HandleTypeDef hadc4;

extern COMP_HandleTypeDef hcomp4;
extern COMP_HandleTypeDef hcomp5;

extern DAC_HandleTypeDef hdac;

extern OPAMP_HandleTypeDef hopamp3;
extern OPAMP_HandleTypeDef hopamp4;

extern RTC_HandleTypeDef hrtc;

extern RTC_TimeTypeDef sTime;
extern RTC_DateTypeDef sDate;

extern uint32_t ticnt, switches;

unsigned int iTx = 0;
extern uint8_t UserTxBufferFS[128];

void lineout(void) {
  UserTxBufferFS[iTx++] = '\n';
  CDC_Transmit_FS(UserTxBufferFS, iTx);
  HAL_Delay(4);
  iTx = 0;
}
void overline(unsigned int spc) {
  if (iTx > sizeof(UserTxBufferFS) - spc - 2) {
    CDC_Transmit_FS(UserTxBufferFS, iTx);
    HAL_Delay(4);
    iTx = 0;
  }
}
HAL_StatusTypeDef stsout(int sts) {
  overline(1);
  switch (sts) {
  case HAL_TIMEOUT: UserTxBufferFS[iTx++] = 't'; break;
  case HAL_BUSY: UserTxBufferFS[iTx++] = 'b'; break;
  case HAL_ERROR: UserTxBufferFS[iTx++] = 'e'; break;
  case HAL_OK: UserTxBufferFS[iTx++] = '.'; break;
  default: UserTxBufferFS[iTx++] = '?';
  }
  return sts;
}
void uintout2(uint8_t val) {
  overline(2);
  UserTxBufferFS[iTx++] = (val / 10) + '0';
  UserTxBufferFS[iTx++] = (val % 10) + '0';
}
void uintout5(uint16_t val) {
  uint16_t v = val;
  overline(5);
  for (int i = iTx + 4; i >= iTx; i--) {
    UserTxBufferFS[i] = (v % 10) + '0';
    v = v / 10;
  }
  iTx += 5;
}
void uintout0x8(uint32_t val) {
  uint32_t v = val;
  overline(8);
  for (int i = iTx + 7; i >= iTx; i--) {
    int dig = v & 0xF;
    UserTxBufferFS[i] = dig + ((dig > 9) ? 'A'-10 : '0');
    v = v >> 4;
  }
  iTx += 8;
}
void strout(char* str) {
  int len = strlen(str);
  overline(len);
  for (int i = 0; i < len; i++) {
    UserTxBufferFS[iTx++] = str[i];
  }
}
void errout(uint8_t* file, uint32_t line) {
  lineout();
  strout((char*)file);
  strout(":");
  uintout5(line);
  strout(": error");
  lineout();
}

/* String received over USB-CDC */
int rxlen = 0;
uint8_t rxbuf[18];
int rdx = 0;
int uintin2(uint8_t* rcvd_buf) {
  if (rdx + 2 >= rxlen) return -1;
  uint8_t chr1 = rcvd_buf[rdx];
  uint8_t chr2 = rcvd_buf[rdx+1];
  if (chr1 < '0' || chr1 > '9') return -1;
  if (chr2 < '0' || chr2 > '9') return -1;
  int num = (chr1 - '0')*10 + (chr2 - '0');
  if (num > 59) return -num;
  rdx = rdx + 2;
  return num;
}

void capture_datime(uint8_t *datime, RTC_TimeTypeDef *sTime, RTC_DateTypeDef *sDate) {
  datime[0] = sDate->Year;
  datime[1] = sDate->Month;
  datime[2] = sDate->Date;
  datime[3] = sTime->Hours;
  datime[4] = sTime->Minutes;
  datime[5] = sTime->Seconds;
}

void spew_datime(uint8_t *datime) {
  strout("20");
  int idx = 0;
  for (; idx < 3; idx++) uintout2(datime[idx]);
  strout("T");
  for (; idx < 6; idx++) uintout2(datime[idx]);
}

uint8_t new_datime[6];
/* capture_mode = -2; download saved readings -> USB */
/* capture_mode = -1; stream readings -> USB */
/* capture_mode = 1; readings -> flash */
/* capture_mode = 2; erase flash or initialization */
int capture_mode = 2;
int reached_temp = 0;
int finished = 0;

void set_date_from_CDC(void) {
  spew_datime(new_datime);
  strout(" -> ");
  stsout(HAL_RTC_SetTime(&hrtc, &sTime, FORMAT_BIN));
  stsout(HAL_RTC_SetDate(&hrtc, &sDate, FORMAT_BIN));
  capture_datime(new_datime, &sTime, &sDate);
  spew_datime(new_datime);
  lineout();
}

void capture_date_from_CDC(uint8_t* rcvd_buf) {
  int num;
  rdx = 0;
  if (20 != uintin2(rcvd_buf)) return;
  HAL_RTC_GetTime(&hrtc, &sTime, FORMAT_BIN);
  HAL_RTC_GetDate(&hrtc, &sDate, FORMAT_BIN);
  capture_datime(new_datime, &sTime, &sDate);
  num = uintin2(rcvd_buf); if (num < 0) return; sDate.Year = num;
  num = uintin2(rcvd_buf); if (num < 0) return; sDate.Month = num;
  num = uintin2(rcvd_buf); if (num < 0) return; sDate.Date = num;
  if ('T' == rcvd_buf[rdx]) rdx++;
  num = uintin2(rcvd_buf); if (num < 0) return; sTime.Hours = num;
  num = uintin2(rcvd_buf); if (num < 0) return; sTime.Minutes = num;
  num = uintin2(rcvd_buf); if (num < 0) return; sTime.Seconds = num;
  /* sDate.WeekDay = RTC_WEEKDAY_MONDAY; */
  sTime.TimeFormat = RTC_HOURFORMAT_24;
  sTime.DayLightSaving = RTC_DAYLIGHTSAVING_NONE;
  sTime.StoreOperation = RTC_STOREOPERATION_RESET;
  if ('D'==rcvd_buf[rdx])
    capture_mode = -2;		/* download */
  else
    capture_mode = -1;		/* stream */
}

/* runs from interrupt */
void acrx(uint8_t* Buf, uint32_t Len) {
  int idx = 0;
 process:
  while (idx < Len && rxlen < 16 && '\n' != Buf[idx] && '\r' != Buf[idx])
    rxbuf[rxlen++] = Buf[idx++];
  if (idx >= Len) return;
  if (rxlen==16 && ('\n'==Buf[idx] || '\r'==Buf[idx])) {
    capture_date_from_CDC(rxbuf);
    rxlen = 0;
  } else
    while (idx < Len && '\n' != Buf[idx] && '\r' != Buf[idx]) idx++;
  while (idx < Len && ( '\n'==Buf[idx] || '\r'==Buf[idx] )) {
    rxlen = 0;
    idx++;
  }
  if (idx==Len) return;
  goto process;
}

unsigned int lidx;
void light_show(uint32_t ons) {
  lidx = (lidx + 1) & 0x03;
  GPIOE->BSRRH = 0xFF00;	/* turn off board leds */
  GPIOE->BSRRL = ((ons >> (lidx * 8)) & 0xFF) << 8;
}

typedef struct {
  uint8_t datime[6];
  uint16_t quants[13];
  /*  0 heatdrive */
  /*  1 heatsense */
  /*  2 heatgnd */
  /*  3 ambientT */
  /*  4 plateT */
  /*  5 backT */
  /*  6 transT */
  /*  7 humidity */
  /*  8 airpressure */
  /*  9 apref */
  /* 10 fancnt */
  /* 11 fanper */
  /* 12 nREPS */
} readingset;

readingset readings;

#define nREPS (16L)

/* Read sequence of ADC inputs multiple times */
void ADC_readn(ADC_HandleTypeDef *hadc, int ist, int ind) {
  for (int reps = 0; reps < nREPS; reps++) {
    HAL_ADC_Start(hadc);
    for (int idx = ist; idx < ind; idx++) {
      HAL_ADC_PollForConversion(hadc, 1);
      readings.quants[idx] += HAL_ADC_GetValue(hadc);
    }
    HAL_ADC_Stop(hadc);
  }
}

extern uint16_t fan_cnt, fan_per, fan_rpm, tarpm, goalrpm;

void take_readings(readingset *rdngs) {
  while(ticnt > 0)
    HAL_RTC_WaitForSynchro(&hrtc);
  HAL_RTC_GetTime(&hrtc, &sTime, FORMAT_BIN);
  HAL_RTC_GetDate(&hrtc, &sDate, FORMAT_BIN);
  GPIOE->BSRRH = 0xFF00;	/* turn off board leds */
  capture_datime(rdngs->datime, &sTime, &sDate);
  memset(rdngs->quants, 0, sizeof(rdngs->quants));
  rdngs->quants[10] = fan_cnt;
  rdngs->quants[11] = fan_per;
  rdngs->quants[12] = nREPS + ((TICSPERS / 10) << 8);
  rdngs->quants[0] = HAL_DAC_GetValue(&hdac, DAC_CHANNEL_1);
  ADC_readn(&hadc1, 1, 3);	/* heatsense, heatgnd */
  GPIOE->BSRRL = GPIO_PIN_6;	/* power on ambient temp sensor */
  HAL_Delay(2);			/* 2.ms delay; settling time is 30.us */
  ADC_readn(&hadc2, 3, 6);	/* ambientT, plateT, backT */
  GPIOE->BSRRH = GPIO_PIN_6;	/* power off ambient temp sensor */
  ADC_readn(&hadc3, 6, 8);	/* transT, humidity */
  ADC_readn(&hadc4, 8, 10);	/* airpressure, apref */
  HAL_ADCEx_Calibration_Start(&hadc1, ADC_SINGLE_ENDED);
  HAL_ADCEx_Calibration_Start(&hadc2, ADC_SINGLE_ENDED);
  HAL_ADCEx_Calibration_Start(&hadc3, ADC_SINGLE_ENDED);
  HAL_ADCEx_Calibration_Start(&hadc4, ADC_SINGLE_ENDED);
}

#define READINGS_BASE 0x08010000 /* FLASH + 64kB */
#define LOGGER_SECS 6144
const FLASH_EraseInitTypeDef pEraseInit = {TYPEERASE_PAGES,
					   READINGS_BASE,
					   96}; /* 192kB out of 256kB */
uint8_t *flash_addr = (uint8_t *)READINGS_BASE;

void erase_flash_data_area(void) {
  uint32_t PageError = 0;
  /* strout("sizeof(readingset)="); uintout2(sizeof(readingset)); */
  stsout(HAL_FLASH_Unlock());
  if (HAL_OK != stsout(HAL_FLASHEx_Erase((FLASH_EraseInitTypeDef*)&pEraseInit,
					 &PageError))) {
    uintout0x8(PageError);
    lineout();
  } else iTx = 0;		/* discard stsouts */
  HAL_FLASH_Lock();
}

HAL_StatusTypeDef store_readings(readingset *rdngs) {
  int scnt = 0;
  HAL_StatusTypeDef sts = HAL_FLASH_Unlock();
  if (HAL_OK != sts) {
    stsout(sts);
    lineout();
    return sts;
  }
  for (scnt = 0; (scnt << 2) < sizeof(readings); scnt++) {
    sts = HAL_FLASH_Program(TYPEPROGRAM_WORD,
			    (uint32_t)flash_addr + (scnt << 2),
			    ((uint32_t *)rdngs)[scnt]);
    if (HAL_OK != sts) {
      stsout(sts);
      uintout2(HAL_FLASH_GetError());
      lineout();
      return sts;
    }
  }
  flash_addr += sizeof(readings);
  return HAL_FLASH_Lock();
}

void spew_readingset(readingset *readings) {
  spew_datime(readings->datime);
  for (int idx = 0; idx < 13; idx++) {
    UserTxBufferFS[iTx++] = '\t';
    uintout5(readings->quants[idx]);
  }
  lineout();
}

void spew_readings(void) {
  readingset *rdngs = (readingset*)READINGS_BASE;
  int cnt = 0;
  while (0==cnt || (0xFF != ( 0xFF & rdngs[cnt].datime[0] ) &&
		    0x00 != ( 0xFF & rdngs[cnt].datime[0] ))) {
    /* 0 < memcmp(rdngs[cnt].datime, rdngs[cnt-1].datime, 6) */
    GPIOE->BSRRL = (cnt & 0xFF) << 8;
    spew_readingset(&rdngs[cnt++]);
    GPIOE->BSRRH = 0xFF00;	/* turn off board leds */
  }
}

void shutdown() {
  HAL_DAC_SetValue(&hdac, DAC_CHANNEL_1, DAC_ALIGN_12B_R, 0);
  HAL_DAC_Start(&hdac, DAC_CHANNEL_1);
  if (1==capture_mode) {
    memset(&readings, 0, sizeof(readingset)); /* EOF */
    store_readings(&readings);
  }
  int dura = 0;
  while (dura < 7) {
    if (0==HAL_GPIO_ReadPin(GPIOA, GPIO_PIN_0)) dura++; else dura = 0;
    light_show(0xFFFFFFFF);
    HAL_Delay(250);
  }
  light_show(0);
}

/* used in log_2() */
uint32_t ush(uint32_t k, int n) {
  if (n < 0) return k >> -n;
  else return k << n;
}

/* used in log_2() */
int ulength(uint32_t k) {
  uint32_t acc = k;
  int len = 0;
  while (0 < acc) { acc = (acc >> 1); len++; }
  return len;
}

/* 24.8 fixnum arithmetic */
/* #define fx8mul(a,b) (((a)*(b))>>8) */
/* #define fx8div(a,b) ((((a)+0L)<<8)/(b)) */
int32_t fx8mul(int32_t a,int32_t b) {
  int32_t prod = a*b;
  return prod >>8;
}
int32_t fx8div(int32_t a,int32_t b) {
  a = a << 8;
  return a / b;
}

/* functions for setting and holding thermistor disk temperature */
/* voltage is (readings.quants[1]-readings.quants[2])/1514 */
/* (define (R_dsk drv E_dsk) */
/*    (/ (* E_dsk (+ R_wiring R_ballast)) */
/* 	 (- (* R_ballast (I_drv drv)) E_dsk))) */
uint32_t fx8R_dsk(uint32_t drv, uint32_t ADV) {
  if (16L > drv) return 8L<<8;
  uint32_t fx8I200 = (256*drv/91); /* 200*I */
  uint32_t fx8E = (ADV<<8)/1514L;
  uint32_t num = fx8E * 202L;
  uint32_t tmp = fx8div(num, fx8I200 - fx8E);
  /* printf("E: %.2f; %.2f/%.2f --> %.2f\n", fx8E/256., num/256., (fx8I200-fx8E)/256., tmp/256.); */
  return tmp;
}

/* return base-2 logarithm in 24.8 fixnum */
uint32_t log_2(uint32_t k) {
  int cnt = 32/(1+ulength(k));
  int pow = 1;
  uint32_t acc = k;
  if (1 >= k) return 0L;
  while (pow < cnt) {acc = acc*k; pow = pow+1;}
  {
    int ilen = ulength(acc);
    uint32_t hibit = 1L << ilen;
    uint32_t tmp = (ilen*(hibit/2L) + acc - hibit);
    return ush(tmp/pow, 9-ilen);
  }
}

/* return base-2 logarithm in 24.8 fixnum */
uint32_t fx8log_2(uint32_t k) {
  uint32_t log0 = log_2(k>>8L);
  if (0L == (k & 0xffL)) return log0;
  {
    uint32_t log1 = log_2((k>>8L)+1);
    return log0 + fx8mul(log1-log0,(k & 0xffL));
  }
}

uint32_t fx8exp2(uint32_t lg2) {
  uint32_t tmp = (lg2>>8);
  uint32_t frac = ((lg2) & 0xffL);
  /* (248-frac/31) is monotonic, but cbrt is not strictly monotonic */
  /* -2.96% < sqrt < +3.05% */
  /* -3.08% < cbrt < +3.04% */
  return ((1L<<tmp)+ush(frac+(248-frac/31),tmp));
}

/* Newton root finder with delta=1/16 */
uint32_t fx8sqrt(uint32_t k) {
  int32_t x = ush(256,(ulength(k))/2-4);
  /* printf("x = %.3f\n", x/256.); */
  int tries=10;
  while(0 < tries--) {
    int32_t eps = fx8mul(x,x)-k;
    if (eps < 0) eps = -eps;
    if (eps < x/64+4) return x;
    x = (x + fx8div(k,x))/2;
    /* printf("x = %.3f\n", x/256.); */
  }
  return 1<<8;
}
/* uint32_t fx8sqrt(uint32_t k) { */
/*   return fx8exp2(fx8log_2(k)/2); */
/* } */
uint32_t fx8sxrt(uint32_t k) {
  return fx8exp2(fx8log_2(k)/6);
}

/* model of thermistor temperature given its resistance */
/* it is used for feedback to stabilize the disk temperature. */
uint32_t fx8T_fit(uint32_t R) {
  /* (+ 7.6 (* 20 (sqrt (log2 (- R 8))))) */
  /* uint32_t log2R = fx8log_2(R-2048L); */
  /* return 1946L+fx8sqrt(400L*log2R); */
  /* (+ 7.4 (* 20 (sqrt (log2 (- R 6.64))))) */
  /* uint32_t log2R = fx8log_2(R-1700L); */
  /* return 1894L+fx8sqrt(400L*log2R); */
  /* (set! T_spread -1.45) (set! T_R 5.2) disk-11 */
  /* uint32_t log2R = fx8log_2(R-1702L); */
  /* return 960L+fx8sqrt(400L*log2R); */
  /* (set! T_spread -2.07) (set! T_R 4.6) disk-11 2025-01-31 */
  /* uint32_t log2R = fx8log_2(R-1708L); */
  /* return 648L+fx8sqrt(400L*log2R); */
  /* (+ T_R T_spread (* 20 (sqrt (log2 (- R (- T_R T_spread)))))) */
  /* uint32_t log2R = fx8log_2(R-1884L); */
  /* return 1731L+fx8sqrt(400L*log2R); */
  /* (set! T_spread -0.22) (set! T_R 7.08) ? */
  /* uint32_t log2R = fx8log_2(R-1884L); */
  /* return 1718L+fx8sqrt(400L*log2R); */
  /* (set! T_R 7.03) (set! T_spread -0.43) */
  uint32_t log2R = fx8log_2(R-1910L);
  return 1690L+fx8sqrt(400L*log2R);
}

uint32_t fx8Nu_F(uint32_t rpm) {
  /*'t model L^{-2}-norm compression with 2 slopes */
  uint32_t Re = (rpm<300L)? 393L*rpm: 393L*300L+361L*(rpm-300L);
  uint32_t Nu = fx8div(fx8mul(152L,Re),(fx8sqrt(Re)+6271L));
  /* return fx8mul(Nu,384L);  /\* with struts, needs more *\/ */
  Nu = fx8mul(294L,Nu);		/* 115% */
  return Nu;
}

uint32_t fx8Ra(uint32_t Tdsk, uint32_t Tamb) {
  return fx8mul((Tdsk-Tamb), 16*fx8div((16*12112L),((Tamb + 69926L))));
}

/* power in mW required at constant temperature Tdsk */
uint32_t P_cnv(uint32_t Tdsk, uint32_t Tamb, uint32_t rpm) {
  uint32_t Nu_N = 211+fx8mul(99,fx8sxrt(fx8Ra(Tdsk,Tamb)));
  Nu_N = fx8mul(Nu_N,Nu_N);
  /* Nu_N = fx8mul(Nu_N,288L);	/\* 12.5% extra for struts *\/ */
  /* if (0L==rpm) return fx8mul(fx8mul(Nu_N,460L),(Tdsk-Tamb)); */
  uint32_t Nu_F = fx8Nu_F(rpm);
  /* L^2 norm */
  uint32_t Nu_N2 = fx8mul(Nu_N,Nu_N);
  uint32_t Nu_F2 = fx8mul(Nu_F,Nu_F);
  uint32_t Nu_2 = fx8sqrt(Nu_N2+Nu_F2);
  uint32_t Nu = (3*Nu_2+2*(Nu_N+Nu_F))/5;
  /* printf("Nu_C = %.1f\n", Nu_F/256.); */
  /*       mW  prec  k     A        L */
  /* (/ (* 1e3 256 26.e-3 528e-6) 7.64e-3) --> 459.9958115183246 */
  uint32_t P = fx8mul(fx8mul(Nu,460L),(Tdsk-Tamb));
  /* printf("P = %.1f\n", P/256.); */
  return P;
}

/* DAIss(Tdsk,Tamb)=18377*sqrt(P_cnv(Tdsk,Tamb)/RfromT(Tdsk))*(1+RfromT(Tdsk)/200) */
/* output is DAI */
uint32_t mW2DAI(uint32_t R, uint32_t P) {
  /* printf("R = %.1f; P = %.1f\n", R/256., P/256.); */
  uint32_t sqrtP = fx8sqrt(P);
  uint32_t sqrtR = fx8sqrt(R);
  uint32_t sqrtPoR = fx8div(sqrtP,sqrtR);
  /* printf("sqrt(P/R)) = %.1f\n", sqrtP/256.); */
  uint32_t I = fx8mul(sqrtPoR,((1L<<8)+fx8div(R, (200L<<8))));
  /* printf("I = %.1f\n", I/256.); */
  I = (72L*I);
  /* printf("I = %.1f\n", I/256.); */
  uint32_t res = (I / 32L);    /* sqrt(1000) ~ 32 */
  /* printf("res = %.1f\n", res/256.); */
  return res;
}

/* P_stp(R)=1e3/KpWs/(1+R/200) */
uint32_t P_stp(uint32_t R, uint32_t KpWs)
{
  uint32_t scl = ((1<<8)+fx8div(R, 200<<8));
  return fx8div(1000*scl, KpWs);
}

/* 407 = 4095/3.3 * 5C * 10mV/C * 210k/32k */
/*  81 = 4095/3.3 * 1C * 10mV/C * 210k/32k */

int logger_main(void) {
  int ambientT0 = 0;		/* initial ambient temperature */
  int overheated = 0;
  uint32_t ptrn = 0xF0F00F0F;
  int32_t mdrv = 0L;		/* 24.8 fixed precision */
  /* prevent limit cycles */
  int32_t mindrv = 768L;	/* 0.056.A ? */
  int32_t drv = 0L;
  /* 24.8 fixed precision */
  int32_t Tbse = 37L<<8;
  int32_t Tinc = 4L<<8;
  int32_t Ttgt = Tbse - Tinc;
  int32_t Trun = 20L<<8;	/* running average of Tfit */
  int32_t Rdsk = 8L<<8;
  uint32_t KpWs = 56L<<8;
  uint32_t cboost = 253L;	/* 24.8 fixed precision of 0.99 */
  uint32_t gain = 1L<<8;
  int stepduration = 100;
  int stepcnt = 6;
  int dura;
  finished = 0;
  HAL_RTC_WaitForSynchro(&hrtc);
  HAL_RTC_GetTime(&hrtc, &sTime, FORMAT_BIN);
  HAL_RTC_GetDate(&hrtc, &sDate, FORMAT_BIN);
  ticnt = 124;		    /* To avoid race with systick fan speed */

  HAL_NVIC_EnableIRQ(COMP4_5_6_IRQn);
  HAL_COMP_Start_IT(&hcomp4);
  HAL_COMP_Start_IT(&hcomp5);

 cmdwait:
  mdrv = 0L;
  drv = 0L;
  dura = 0;
  ambientT0 = 0;		/* initial ambient temperature */
  reached_temp = 0;
  capture_mode = 0;

  while (0==capture_mode && (finished || 0==HAL_GPIO_ReadPin(GPIOA, GPIO_PIN_0))) {
    light_show(overheated ? 0xFFFF0000 :
	       (finished ? 0x28448210 : 0x01824428));
    /* light_show(0x83C64401); */
    HAL_Delay(250);
  }
  if (capture_mode < 0) {
    set_date_from_CDC();
    if (-2==capture_mode) {
      spew_readings();
      capture_mode = 0;
      /* finished = 0; */
      goto cmdwait;
    }
  } else {			/* blue button has been pushed */
    capture_mode = 2;
  }

  light_show(0xFFFFFFFF);
  if (2==capture_mode) {
    erase_flash_data_area();
    light_show(0x11224488);
    while(ticnt > 0)
      HAL_RTC_WaitForSynchro(&hrtc); /* so that fan count is reset */
    capture_mode = 1;
    for (int secnt = 0; secnt < 10; secnt++) {
      HAL_Delay(1000);
      light_show(0x11224488);
    }
    HAL_RTC_GetTime(&hrtc, &sTime, FORMAT_BIN);
    HAL_RTC_GetDate(&hrtc, &sDate, FORMAT_BIN);
  }
  light_show(0);
  HAL_Delay(10);
  HAL_ADCEx_Calibration_Start(&hadc1, ADC_SINGLE_ENDED);
  HAL_ADCEx_Calibration_Start(&hadc2, ADC_SINGLE_ENDED);
  HAL_ADCEx_Calibration_Start(&hadc3, ADC_SINGLE_ENDED);
  HAL_ADCEx_Calibration_Start(&hadc4, ADC_SINGLE_ENDED);
  HAL_OPAMP_SelfCalibrate(&hopamp3);
  HAL_OPAMP_SelfCalibrate(&hopamp4);
  HAL_OPAMP_Start(&hopamp3);
  HAL_OPAMP_Start(&hopamp4);
  HAL_Delay(10);
  dura = 0;
  ptrn = 0x0;
  /* plate runs 102 minutes */
  /* disk capture runs 11:40 minutes */
  /* disk calibration runs 23:20 minutes */
  for (int secnt = 0; !overheated; (finished ? 0 : (secnt=secnt+1))) {
    int blue_button = HAL_GPIO_ReadPin(GPIOA, GPIO_PIN_0);
    /* Stop if blue button is held for more than 3 seconds */
    if (dura >= 2 || secnt == LOGGER_SECS - 2) {
      shutdown();
      if (1==capture_mode) {
	finished = 1;
      }
      goto cmdwait;
    }
    if (0==blue_button) dura = 0; else dura++;
    take_readings(&readings);
    light_show((readings.quants[2] & 0xFFFF) > 2000*nREPS ? 0xA000A000 : ptrn);
    if (1==capture_mode) store_readings(&readings);

    /* overheating 100C * 10mV/C * (4096/3.3V) * nREPS --> 1241 * nREPS */
    /* overheating 125C * 10mV/C * (4096/3.3V) * nREPS --> 1551 * nREPS */
    overheated = readings.quants[6] > 1551 * nREPS;
    /* will exit after this iteration */

    if (0 == ambientT0) ambientT0 = readings.quants[3] & 0xFFFF;
    int32_t senseV0 = (readings.quants[1]-readings.quants[2]);
    if (senseV0 < 0L) senseV0 = -senseV0;

    /* Here are the (time) programs for heating: */

    if (0 == HAL_GPIO_ReadPin(GPIOB, GPIO_PIN_10)) {
      /* thermistor disk programs */

      /* mdrv, T*, and Rdsk are in 24.8 bit format */
      switch ((switches/1000)%5) {
      case 0:		 /* step current through 100.0.Ohm resistor */
	Ttgt = 0L;
	reached_temp = 1;
	if (40 <= secnt) {
	  if (1==capture_mode) {
	    finished = 1;
	  }
	  drv = 0L;
	  mdrv = 0L;
	  Ttgt = 0L;
	  shutdown();
	  goto cmdwait;
	} else if (secnt < 7) drv = 0;
	else drv = (4095L) >> (3 - ((((secnt+1)/8)-1) % 4));
	break;
      case 1: /* measure with two LM37s clamped to thermistor disk */
	KpWs = 84L<<8;		/* assume 150% of thermal mass (84) */
	/* assume natural convection area is 2.35 x bare disk (602) */
	cboost = 602L;
	/* gain = 1L<<8; */
	/* Tinc = 4L<<8; */
	if (1L > secnt) Ttgt = Tbse-Tinc;
	/* prevent limit cycles */
	/* mindrv = 1024L;		/\* 0.056.A *\/ */
	/* stepduration = 100; */
	stepcnt = 6L;
      case 2:
	if ((1L > secnt) && (2==(switches/1000)%5)) {
	  /* reached_temp = 1; */
	  /* stepduration = 75; */
	  /* secnt = stepduration; */
	  /* Tinc = 4L<<8; */
	  Ttgt = Tbse-Tinc;
	  Trun = 20L<<8;	/* running average of Tfit */
	  mdrv = (4095L);
	  drv = mdrv;
	  ptrn = 0x20022002;
	}
      default: /* measure bare thermistor disk */
	/* contin: */
	Rdsk = fx8R_dsk(drv,senseV0);
	if (secnt+((1==capture_mode)?1:0) >=
	    stepduration*((1==(switches/1000)%5) ? (2*stepcnt+2) : (stepcnt+1))) {
	  if (1==capture_mode) {
	    finished = 1;
	  }
	  drv = 0L;
	  mdrv = 0L;
	  Ttgt = 0L;
	  shutdown();
	  goto cmdwait;
	} else if (1L > secnt) {
	  Trun = 20L<<8;	/* running average of Tfit */
	  mdrv = (4095L);
	  drv = mdrv;
	  ptrn = 0x20022002;
	} else if (2432L > Rdsk) { /* not in monotonic range 9.5.Ohms  */
	  mdrv = (4095L);
	  drv = mdrv;
	  if (0 == (secnt % stepduration) && (0 != ((switches/1000)%5))) {
	    /* ramp up temperature in 5.C intervals at stepduration intervals */
	    Ttgt = Ttgt + Tinc;
	  }
	} else {
	  if (0 == (secnt % stepduration) && (0 != ((switches/1000)%5))) {
	    /* ramp up temperature in 4.C steps at stepduration intervals */
	    if ((1==(switches/1000)%5) && (secnt >= stepduration*(2+stepcnt))) {
	      Ttgt = Ttgt - Tinc;
	    } else {
	      Ttgt = Ttgt + Tinc;
	    }
	  }

	  uint32_t Tbump = MIN(59L<<8,MAX(Tbse,Ttgt));
	  uint32_t Tfit = fx8T_fit(Rdsk);
	  ptrn = (Tfit>Trun) ? 0x20022002 : 0x10011001;
	  Trun = (2*Trun+Tfit)/3;
	  uint32_t Tamb = 177L + 32L*(readings.quants[3] & 0xFFFF)/165L;
	  /* uint32_t Pnow =    fx8mul(cboost, P_cnv(Trun, Tamb, (1==(switches/1000)%5)?0:fan_rpm)); */
	  uint32_t Pexpect = fx8mul(cboost, P_cnv(Tbump, Tamb, MIN(tarpm,goalrpm))); /* (1==(switches/1000)%5)?0:fan_rpm */
	  uint32_t Pstep = P_stp(Rdsk, KpWs);
	  /* uint32_t ndrv = mW2DAI(Rdsk, MAX(0L,Pexpect + fx8mul(Pstep,fx8div((Tbump-Trun)*gain, fx8sqrt(Pexpect))))); */
	  uint32_t ndrv = mW2DAI(Rdsk, MAX(0L,Pexpect + fx8mul(Pstep,fx8mul((Tbump-Trun),gain))));

	  mdrv = ndrv;
	  mdrv = MAX(mindrv, mdrv);
	  mdrv = MIN(4095L, mdrv);
	  drv = mdrv;

	  if (2==(switches/1000)%5) {
	    if ((secnt >= stepduration))
	      reached_temp = (((secnt % stepduration) > (stepduration/4) || secnt <= 3*stepduration) ? 1 : 0);
	  } else {
	    if ((1==(switches/1000)%5) && (secnt >= stepduration*(2+stepcnt))) {
	      reached_temp = 1;
	    }
	  }
	}
	break;
      }
    }
    else {
      /* rough plate program */

      /* Threshold is high digit (of switches) modulo 5 times 5.C above ambient */
      long plateT = readings.quants[4] & 0xFFFF;
      /* long backT = readings.quants[5] & 0xFFFF; */
      if ((plateT >= ((switches/1000)%5)*407*nREPS + ambientT0 + nREPS/2
	   && !reached_temp)
	  || (1 != capture_mode))
	reached_temp = 1;
      if (reached_temp || secnt >= 3750) {
	drv = 0L;	/* cool down */
	ptrn = 0x10011001;
      }
      else if (secnt >= 150) {
	drv = 4095L; /* full on up to 60 minutes */
	ptrn = 0x20022002;
      }
      else {
	drv = 0L;		     /* heater off 1 minute */
	ptrn = 0x80088008;
      }
    }
    if (overheated) drv = 0L;
    drv = MAX(0L, drv);
    drv = MIN(4095L, drv);
    HAL_DAC_SetValue(&hdac, DAC_CHANNEL_1, DAC_ALIGN_12B_R, drv);
    HAL_DAC_Start(&hdac, DAC_CHANNEL_1);
    if (-1==capture_mode) spew_readingset(&readings);
    /* spew_readingset((readingset*)(flash_addr - sizeof(readingset))); */
  }
  if (overheated) {
    shutdown();
  }
  goto cmdwait;
}
